#ifndef RESISTIVE_TOUCH_CONTROLLER_HPP
#define RESISTIVE_TOUCH_CONTROLLER_HPP

//-- Macros for calibration function
#define X_T1 t[0][0]
#define X_T2 t[1][0]
#define X_T3 t[2][0]

#define Y_T1 t[0][1]
#define Y_T2 t[1][1]
#define Y_T3 t[2][1]

#define X_D1 d[0][0]
#define X_D2 d[1][0]
#define X_D3 d[2][0]

#define Y_D1 d[0][1]
#define Y_D2 d[1][1]
#define Y_D3 d[2][1]

#include "mbed.h"

/**
  * A library to get xy-coordinates from a 5-wire resistive touchc screen
  *
  * @author CA Bezuidenhout
  */
class ResistiveTouchController
{
public:
  /**
   * @param ur : Upper right pin of screen (Digital)
   * @param lr : Lower right pin of screen (Digital)
   * @param s : Wiper pin of screen (Analog)
   * @param ul : Upper left pin of screen (Digital)
   * @param ll : Lower left pin of screen (Digital)
   * @param num_of_samples : Number of samples used to calculate average analog value (default 5)
   * @param settle : Time in seconds required for the electric field to settle (default 2ms)
   */
  ResistiveTouchController(PinName ur, PinName lr, PinName s, PinName ul, PinName ll, int num_of_samples = 5, float settle =  0.002);

  /**
   Determines if screen is touched.
   *
   * @returns 1 if touch is detected, 0 if not.
   */
  int TouchDetected();

  /**
   * Measured the analog values for x and y coordinates
   */
  void Measure();

  /**
   * Get the raw analog value of the x coordinate
   *
   * @returns Raw analog value of x coordinate
   */
  float RawX();

  /**
   * Get the raw analog value of the y coordinate
   *
   * @returns Raw analog value of y coordinate
   */
  float RawY();

  /**
   *Get the digital value of the x coordinate, calculated using \f$x = Ax_t + By_t+C\f$
   *
   * @returns Digital value of x coordinate
   */
  float X();

  /**
   * Get the digital value of the y coordinate, calculated using \f$y = Dx_t + Ey_t+F\f$
   *
   * @returns Digital value of y coordinate
   */
  float Y();

  /**
   * Calibrates the screen from 3 points.
   *
   * @param t[3][2] : Array containing the 3 raw data points
   *  <ul>
   *  <li>t[0][0] = Raw x value at 90% of x</li>
   *  <li>t[1][0] = Raw x value at 50% of x</li>
   *  <li>t[2][0] = Raw x value at 10% of x</li>
   *  </ul>
   *  <ul>
   *  <li>t[0][1] = Raw y value at 50% of y</li>
   *  <li>t[1][1] = Raw y value at 90% of y</li>
   *  <li>t[2][1] = Raw y value at 10% of y</li>
   *  </ul>
   * -------------------------------------------------------
   * @param d[3][2] : Array containing the 3 digital points
   * <ul>
   * <li>d[0][0] = Digital x value at 90% of x</li>
   * <li>d[1][0] = Digital x value at 50% of x</li>
   * <li>d[2][0] = Digital x value at 10% of x</li>
   * </ul>
   * <ul>
   * <li>_d[0][1] = Digital y value at 50% of y</li>
   * <li>_d[1][1] = Digital y value at 90% of y</li>
   * <li>_d[2][1] = Digital y value at 10% of y</li>
   * </ul>
   */
  void Calibrate(float t[3][2], float d[3][2]);


  /**
   * Set the settle time of the screen
   *
   * @param settle : Time in seconds required for the electric field to settle (default 2ms)
   */
  void SetSettleTime(float settle);

private:
  DigitalInOut _UR;
  DigitalInOut _LR;
  PinName _S;
  DigitalInOut _UL;
  DigitalInOut _LL;
  Timer _delayTimer;

  float _settle;

  static const char DETECT_MODE = 1;
  static const char X_MODE = 2;
  static const char Y_MODE = 3;

  float A,B,C,D,E,F;

  float _xt, _yt;

  int _samples;

  void ConfigPins(char mode);
  float MeasureS();
  void Delay(float time);
};
#endif
