#include "ResistiveTouchController.hpp"

ResistiveTouchController::ResistiveTouchController(PinName ur, PinName lr, PinName s, PinName ul, PinName ll, int num_of_samples, float settle) :
  _UR(ur),
  _LR(lr),
  _UL(ul),
  _LL(ll)
{
  _UR.output();
  _UR.mode(PullNone);

  _LR.output();
  _LR.mode(PullNone);

  _UL.output();
  _UL.mode(PullNone);

  _LL.output();
  _LL.mode(PullNone);

  _S = s;

  _samples = num_of_samples;
  _settle = settle;

  //-- Change these values if needed for desired screen
  A = 1660.540541f;
  B = 20.5005005f;
  C = -334.0556557f;
  D = 0.0f;
  E = 1517.037037f;
  F = -246.5185185;
}

int ResistiveTouchController::TouchDetected()
{
  DigitalIn S(_S, PullUp);
  ConfigPins(DETECT_MODE);

  return ( !S.read() );
}

void ResistiveTouchController::Measure()
{
    ConfigPins(X_MODE);
    _xt = MeasureS();

    ConfigPins(Y_MODE);
    _yt = MeasureS();
}

float ResistiveTouchController::RawX()
{
  return _xt;
}

float ResistiveTouchController::RawY()
{
  return _yt;
}

float ResistiveTouchController::X()
{
  return (A*_xt+B*_yt+C);
}

float ResistiveTouchController::Y()
{
  return (D*_xt+E*_yt+F);
}

void ResistiveTouchController::Calibrate(float t[3][2], float d[3][2])
{
  A = (X_D1*(Y_T2-Y_T3)+X_D2*(Y_T3-Y_T1)+X_D3*(Y_T1-Y_T2))/(X_T1*(Y_T2-Y_T3)+X_T2*(Y_T3-Y_T1)+X_T3*(Y_T1-Y_T2));
  B = (A*(X_T3-X_T2)+X_D2-X_D3)/(Y_T2-Y_T3);
  C = X_D3-A*X_T3-B*Y_T3;
  D = (Y_D1*(Y_T2-Y_T3)+Y_D2*(Y_T3-Y_T1)+Y_D3*(Y_T1-Y_T2))/(X_T1*(Y_T2-Y_T3)+X_T2*(Y_T3-Y_T1)+X_T3*(Y_T1-Y_T2));
  E = (D*(X_T3-X_T2)+Y_D2-Y_D3)/(Y_T2-Y_T3);
  F = Y_D3-D*X_T3-E*Y_T3;
}

void ResistiveTouchController::SetSettleTime(float settle)
{
  _settle = settle;
}

void ResistiveTouchController::ConfigPins(char mode)
{
  switch (mode)
  {
    case DETECT_MODE:
      _UR.write(0);
      _UL.write(0);
      _LR.write(0);
      _LL.write(0);
      break;
    case X_MODE:
      _UR.write(1);
      _UL.write(0);
      _LR.write(1);
      _LL.write(0);
      break;
    case Y_MODE:
      _UR.write(1);
      _UL.write(1);
      _LR.write(0);
      _LL.write(0);
      break;
  }

  Delay(_settle);
}

void ResistiveTouchController::Delay(float time)
{
  _delayTimer.start();

  while( _delayTimer.read() < time );

  _delayTimer.stop();
  _delayTimer.reset();
}

float ResistiveTouchController::MeasureS()
{
  AnalogIn S(_S);

  float measured_val = 0.0f;

  for( int i =0 ; i < _samples ; i++)
  {
    measured_val += S.read();
  }

  return ( measured_val/_samples);
}