#ifndef _MOTOR_DRIVER_HPP_
#define _MOTOR_DRIVER_HPP_

#include "mbed.h"

/**
*  @bref  モーターを回すライブラリ
*  @author Gaku MATSUMOTO
*  @date 2017/02/18
*/
class motor_driver{
  
  /**
     @param r_f 右の前進ピン
     @param r_b 右の後進ピン
     @param l_f 左の前進ピン
     @param l_b 左の後進ピン
  */
  public:
    motor_driver(PinName r_f, PinName r_b, PinName l_f, PinName l_b, PinName r_st, PinName l_st);
    
  private:
    
    PwmOut RF;
    PwmOut RB;
    PwmOut LF;
    PwmOut LB;
    DigitalOut Rst;
    DigitalOut Lst;
    
  public:
    /**
    *   @bref  引数のデューティー比で前進する．
    *   @param duty デューティー比
    *   @note  引数なしで全力前進
    */
    void forward(float duty = 1.0f);
    
    /**
    *  @bref  引数のデューティー比で後進する．
    *  @param  duty  デューティー比
    *  @note  引数無しで全力後進
    */
    void back(float duty = 1.0f);
    
    /**
    *  @bref  指定した比率で左右のタイヤを回し，右折します．
    *  @param  左右のタイヤへのpwm比
    */
    void right(float percentage = 0.5f);
    
    /**
    *  @bref  指定した比率で左右のタイヤを回し，左折します．
    *  @param  左右のタイヤへのpwm比
    */
    void left(float percentage = 0.5f);
    
    /**
    *  @bref  指定したデューティー比で右に超信地旋回する．
    *  @param  duty  デューティー比
    */
    void pivot_right(float duty = 1.0f);
    
    /**
    *  @bref  指定したデューティー比で左に超信地旋回する．
    *  @param  duty  デューティー比
    */
    void pivot_left(float duty = 1.0f);
    
    /**
    *  @bref  左右のモーターを停止させる．
    *  @note  ブレーキではない
    */
    void stop();
    
    /**
    *  @bref  モータードライバーを省電力待機モードにする．
    *  @note  基本的に走っていないときはこれを実行したほうがよいかと
    */
    void standBy();  
    
    /**
    *  @bref  ブレーキをかける
    *  @note  モーター的にはあまりよくないようです
    */
    void brake();
};

motor_driver::motor_driver(PinName r_f, PinName r_b, PinName l_f, PinName l_b, PinName r_st, PinName l_st) : RF(r_f), RB(r_b), LF(l_f), LB(l_b), Rst(r_st), Lst(l_st){

    RF = 0;
    RB = 0;
    LF = 0;
    LB = 0;
    Rst = 0;
    Lst = 0;
}

inline void motor_driver::forward(float duty){
    RF = duty;
    RB = 0.0f;
    LF = duty;
    LB = 0.0f;      
    Rst = 1;
    Lst = 1;
    return;
}

inline void motor_driver::back(float duty){
    RF = 0.0f;
    RB = duty;
    LF = 0.0f;
    LB = duty;
    Rst = 1;
    Lst = 1;
    return;   
}

inline void motor_driver::right(float percentage){
    //右を弱める
    RF = percentage;
    RB = 0.0f;
    LF = 1.0f;
    LB = 0.0f;
    Rst = 1;
    Lst = 1;
    return;
}

inline void motor_driver::left(float percentage){
    //左を弱める
    RF = 1.0f;
    RB = 0.0f;
    LF = percentage;
    LB = 0.0f;
    Rst = 1;
    Lst = 1;
    return;
}

inline void motor_driver::pivot_right(float duty){
    RF = 0.0f;
    RB = duty;
    LF = duty;
    LB = 0.0f;
    Rst = 1;
    Lst = 1;
    return;
}

inline void motor_driver::pivot_left(float duty){
    RF = duty;
    RB = 0.0f;
    LF = 0.0f;
    LB = duty;
    Rst = 1;
    Lst = 1;
    return;
}

inline void motor_driver::stop(){
    RF = 0.0f;
    RB = 0.0f;
    LF = 0.0f;
    LB = 0.0f;
    Rst = 1;
    Lst = 1;
    return;
}

inline void motor_driver::standBy(){
    Rst = 0;
    Lst = 0;
    RF = 0.0f;
    RB = 0.0f;
    LF = 0.0f;
    LB = 0.0f;
    return;
}

inline void motor_driver::brake(){
    RF = 1;
    RB = 1;
    LF = 1;
    LB = 1;
    Rst = 1;
    Lst = 1;   
}
#endif

