/*******************************************************************************
 *  DHT Library for  Digital-output Humidity and Temperature sensors
 *
 *  Works with DHT11, DHT22
 *             SEN11301P,  Grove - Temperature&Humidity Sensor     (Seeed Studio)
 *             SEN51035P,  Grove - Temperature&Humidity Sensor Pro (Seeed Studio)
 *             AM2302   ,  temperature-humidity sensor
 *             HM2303   ,  Digital-output humidity and temperature sensor
 *
 *  Copyright (C) Wim De Roeve
 *                based on DHT22 sensor library by HO WING KIT
 *                Arduino DHT11 library
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documnetation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to  whom the Software is
 * furished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS OR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * /////////////////////////////////////////////////////////////////////////////
 *
 * Used by Giorgos Tsapparellas for Internet of Things (IoT) smart monitoring
 * device for agriculture using LoRaWAN technology.
 * 
 * Sensor Used: DHT11 - Digital-output Humidity and Temperature sensor.
 *
 * Date of issued copy: 10 January 2018
 *
 * Modifications: 
 * - No external modifications of the existing "AS IT IS" software.
 * - Added some external comments for meeting good principles of 
 *   source code re-usability.   
 ******************************************************************************/

#include "DHT.h"

#define DHT_DATA_BIT_COUNT 40 // Data bit count set to 40

/* Uncomment this line for configuring pc variable for UART usage. */
//Serial pc(USBTX, USBRX); // UART tranmitter(tx) and receiver(rx).

/* 
* DHT constructor.
* Input parameters: PinName pin
*                   eType DHTtype
*
*/ 
DHT::DHT(PinName pin, eType DHTtype)
{
    
    _pin = pin;
    _DHTtype = DHTtype;
    _firsttime = true;
    
} // end of DHT constructor. 

/* 
* Null DHT destructor.
*
*/ 
DHT::~DHT()
{
    
} // end of DHT destructor. 

/* 
* stall DHT function of type eError.
*
* Input parameters: DigitalInOut io
*                   int const level
*                   int const max_time
*
* Return: ERROR_NO_PATIENCE or ERROR_NONE
*/ 
eError DHT::stall(DigitalInOut &io, int const level, int const max_time)
{
    
    int cnt = 0;
    while (level == io) {
        if (cnt > max_time) {
            return ERROR_NO_PATIENCE;
        }
        cnt++;
        wait_us(1);
    }
    return ERROR_NONE;
    
} // end of stall function.

/* 
* readData DHT function of type eError.
*
* Input parameters: None
*
* Return: Either BUS_BUSY, ERROR_NOT_PRESENT, ERROR_SYNC_TIMEOUT, 
*         ERROR_NO_PATIENCE, ERROR_DATA_TIMEOUT or ERROR_CHECKSUM.     
*/ 
eError DHT::readData()
{
    uint8_t i = 0, j = 0, b = 0, data_valid = 0;
    uint32_t bit_value[DHT_DATA_BIT_COUNT] = {0};

    eError err = ERROR_NONE;
    time_t currentTime = time(NULL);

    DigitalInOut DHT_io(_pin);

    // I/O must be in high state to start
    if (ERROR_NONE != stall(DHT_io, 0, 250)) {
        return BUS_BUSY;
    }

    // Start the transfer
    DHT_io.output();
    DHT_io = 0;
    wait_ms(18);
    DHT_io = 1;
    wait_us(30);
    DHT_io.input();
    
    // Wait until the sensor grabs the bus
    if (ERROR_NONE != stall(DHT_io, 1, 100)) {
        return ERROR_NOT_PRESENT;
    }
    // Sensor should signal low 80us and then high 80us
    if (ERROR_NONE != stall(DHT_io, 0, 100)) {
        return ERROR_SYNC_TIMEOUT;
    }
    if (ERROR_NONE != stall(DHT_io, 1, 100)) {
        return ERROR_NO_PATIENCE;
    }
    // Capture the data
    for (i = 0; i < 5; i++) {
        for (j = 0; j < 8; j++) {
            if (ERROR_NONE != stall(DHT_io, 0, 75)) {
                return ERROR_DATA_TIMEOUT;
            }
            // Logic 0 is 28us max, 1 is 70us
            wait_us(40);
            bit_value[i*8+j] = DHT_io;
            if (ERROR_NONE != stall(DHT_io, 1, 50)) {
                return ERROR_DATA_TIMEOUT;
            }
        }
    }
    // Store the data
    for (i = 0; i < 5; i++) {
        b=0;
        for (j=0; j<8; j++) {
            if (bit_value[i*8+j] == 1) {
                b |= (1 << (7-j));
            }
        }
        DHT_data[i]=b;
    }

    // Uncomment to see the checksum error if it exists
    // pc.printf(" 0x%02x + 0x%02x + 0x%02x + 0x%02x = 0x%02x \n", DHT_data[0],
    // DHT_data[1], DHT_data[2], DHT_data[3], DHT_data[4]);
    data_valid = DHT_data[0] + DHT_data[1] + DHT_data[2] + DHT_data[3];
    if (DHT_data[4] == data_valid) {
        _lastReadTime = currentTime;
        _lastTemperature = CalcTemperature();
        _lastHumidity = CalcHumidity();

    } else {
        err = ERROR_CHECKSUM;
    }

    return err;
    
} // end of readData function.

/* 
* CalcTemperature DHT function of type float.
*
* Input parameters: None
*
* Return: float temperature value.     
*/ 
float DHT::CalcTemperature()
{
    
    int v;

    switch (_DHTtype) {
        case DHT11:
            v = DHT_data[2];
            return float(v);
        case DHT22:
            v = DHT_data[2] & 0x7F;
            v *= 256;
            v += DHT_data[3];
            v /= 10;
            if (DHT_data[2] & 0x80)
                v *= -1;
            return float(v);
    }
    return 0;
    
} // end of CalcTemperature function.

/*
* ReadHumidity DHT function of type float.
*
* Input parameters: None
*
* Return: float last humidity value.     
*/ 
float DHT::ReadHumidity()
{
    return _lastHumidity;
} // end of ReadHumidity function.

/*
* ConvertCelsiustoFarenheit DHT function of type float.
*
* Input parameters: float const celsius
*
* Return: float farenheit temperature value.     
*/ 
float DHT::ConvertCelciustoFarenheit(float const celsius)
{
    return celsius * 9 / 5 + 32;
} // end of ConvertCelciustoFarenheit function.

/*
* ConvertCelsiustoKelvin DHT function of type float.
*
* Input parameters: float const celsius
*
* Return: float kelvin temperature value.     
*/ 
float DHT::ConvertCelciustoKelvin(float const celsius)
{
    return celsius + 273.15f;
} // end of ConvertCelciusKelvin function.

/*
* CalcdewPoint DHT function of type float.
*
* Input parameters: float const celsius
*                   float const humidity
*
* Return: dewPoint.     
* Reference: http://wahiduddin.net/calc/density_algorithms.htm
*/ 
float DHT::CalcdewPoint(float const celsius, float const humidity)
{
    float A0= 373.15f/(273.15f + celsius);
    float SUM = -7.90298 * (A0-1);
    SUM += 5.02808f * log10(A0);
    SUM += -1.3816e-7 * (pow(10, (11.344f*(1-1/A0)))-1) ;
    SUM += 8.1328e-3 * (pow(10,(-3.49149*(A0-1)))-1) ;
    SUM += log10(1013.246);
    float VP = pow(10, SUM-3) * humidity;
    float T = log(VP/0.61078f);   // temp var
    return (241.88f * T) / (17.558f-T);
    
} // end of CalcdewPoint function.

/*
* CalcdewPointFast DHT function of type float.
*
* Input parameters: float const celsius
*                   float const humidity
*
* Return: 5x faster dewPoint.     
* Reference: http://en.wikipedia.org/wiki/Dew_point
*/ 
float DHT::CalcdewPointFast(float const celsius, float const humidity)
{
    float a = 17.271;
    float b = 237.7;
    float temp = (a * celsius) / (b + celsius) + log(humidity/100);
    float Td = (b * temp) / (a - temp);
    return Td;
    
} // end of CalcdewPointFast function 

/*
* ReadTemperature DHT function of type float.
*
* Input parameters: eScale Scale
*
* Return: float scaled temperature value.     
* Reference: http://en.wikipedia.org/wiki/Dew_point
*/ 
float DHT::ReadTemperature(eScale Scale)
{
    if (Scale == FARENHEIT)
        return ConvertCelciustoFarenheit(_lastTemperature);
    else if (Scale == KELVIN)
        return ConvertCelciustoKelvin(_lastTemperature);
    else
        return _lastTemperature;
} // end of ReadTemperature function.

/*
* Calc DHT function of type float.
*
* Input parameters: None
*
* Return: float humidity value.     
*/ 
float DHT::CalcHumidity()
{
    int v;

    switch (_DHTtype) {
        case DHT11:
            v = DHT_data[0];
            return float(v);
        case DHT22:
            v = DHT_data[0];
            v *= 256;
            v += DHT_data[1];
            v /= 10;
            return float(v);
    }
    return 0;
} // end of CalcHumidity function. 