#ifndef MBED_TRAIN_H
#define MBED_TRAIN_H

#include "mbed.h"
#include "rtos.h"

class Train {
    
public:
/*Class train
* Based on PWM out pin, and two seperate directions pins. 
* Original aim was to use L293 chip, as H-bridge driver, to drive model
* railroad with 12V
* Pin fwd = forward motion
* Pin rwd = reverse motion
* Pin pwm = pwm between 0 and 1 to set speed of train
*/

/*
 * Example:
 * @code
 *
 * #include "mbed.h"
 * #include "rtos.h"
 * #include "train.h"
 * /////////////////////////Outputs///////////////////////////
 * //Motor driver (train)
 * //  Train train1 (LED3, LED2, LED1); //KL25Z LEDs can be used to simulate motor pins
 * Train train1 (PTC5, PTC7, PTD1); //tested on K64F 
 *  
 * //DEbug serial port, to demonstrate GetDirection and GetSpeed callback
 * Serial pc(USBTX, USBRX);
 *
 * //Main loop
 * int main(){
 * //enable train1 to update speed, and wait for commands
 *  train1.Enable(1);
 *
 *  // continously change direction and accelerate. 
 *  // Changing direction will automatically brake the train to 0 speed.
 *   while(1) {
 *       // SetDirection determines direction, 0 = stop, 1 = forward, 2 = reverse
 *       train1.SetDirection(1);   
 *             wait(2);
 *       train1.SetSpeed(71);
 *             wait(2);
 *       pc.printf("Set direction: %d \r\n", train1.GetDirection());
 *       pc.printf("Set Speed: %d \r\n", train1.GetSpeed());
 *             wait(2);
 *       train1.SetDirection(2);
 *             wait(2);
 *       train1.SetSpeed(56);          
 *             wait(2);
 *       pc.printf("Set direction: %d \r\n", train1.GetDirection());
 *        pc.printf("Set Speed: %d \r\n", train1.GetSpeed());
 *   }
 * }
 * @endcode
 */

//Create Train Class
    Train(PinName fwd, PinName rwd, PinName pwm);
    
// AccConvert (make sure acceleration curve stays between 1-4 seconds)
    int AccConvert (int _acc);
// SetSpeed to value (value between 0 and 100), Train will automaticatly accelerate/brak to set speed    
    void SetSpeed(int trainSpeed);
// Update loop, called by interupt, check for speed changes and apply them
    void Update();
// Update loop, check if speed is 0 before switching direction
    void UpdateDir();
// Init train, now train can receive commands
    void Enable(int _acc);
// Disable train
    void Disable();  
// Direct stop. Abrubtly stops train, setting speed and direction to 0. 
// When needing braking, use train.SetSpeed(0);
    void DirectStop();
// Set new direction to change direction of train. Train will brake to 0, before switching
    void SetDirection(int _dir);   // 0 = standstill, 1 = forward, 2 = reverse

// Get the last set direction    
    int GetDirection();
// Get the last set Speed
    int GetSpeed();
    
//Variables:
    int acc;
    int CurrentDirection;
    int CurrentTrainSpeed;
    int NewTrainSpeed;
    int NewDirection;
 
 
private:  
//Motor driver (train)
DigitalOut  _fwd;   // forward motion
DigitalOut  _rwd;   // rearward motion
PwmOut      _pwm;   // Speed setting with pwm
float       _current_speed;
Ticker t_SetSpeed;
Ticker t_SetDir;
};
 
#endif