#include "dot_util.h"
#include "RadioEvent.h"
#include "LD100_util.h"
 
#if ACTIVE_EXAMPLE == LD100_EXAMPLE

/////////////////////////////////////////////////////////////////////////////
// -------------------- DOT LIBRARY REQUIRED ------------------------------//
// * Because these example programs can be used for both mDot and xDot     //
//     devices, the LoRa stack is not included. The libmDot library should //
//     be imported if building for mDot devices. The libxDot library       //
//     should be imported if building for xDot devices.                    //
// * https://developer.mbed.org/teams/MultiTech/code/libmDot-dev-mbed5/    //
// * https://developer.mbed.org/teams/MultiTech/code/libmDot-mbed5/        //
// * https://developer.mbed.org/teams/MultiTech/code/libxDot-dev-mbed5/    //
// * https://developer.mbed.org/teams/MultiTech/code/libxDot-mbed5/        //
/////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////
// * these options must match the settings on your gateway //
// * edit their values to match your configuration         //
// * frequency sub band is only relevant for the 915 bands //
// * either the network name and passphrase can be used or //
//     the network ID (8 bytes) and KEY (16 bytes)         //
/////////////////////////////////////////////////////////////
static std::string network_name = "MTCDT-18446018";
static std::string network_passphrase = "jointech_passphrase";
static uint8_t network_id[] = { 0x57, 0xD7, 0x64, 0xA4, 0x0C, 0xFE, 0xB1, 0xDF };
static uint8_t network_key[] = { 0x63, 0x81, 0x87, 0x8D, 0x11, 0x45, 0x88, 0x13, 0x30, 0xCF, 0xCF, 0xBC, 0x1D, 0x48, 0xA8, 0xEC };
static uint8_t frequency_sub_band = 2;
static lora::NetworkType network_type = lora::PUBLIC_LORAWAN;
static uint8_t join_delay = 5;
static uint8_t ack = 0;
static bool adr = true;

// deepsleep consumes slightly less current than sleep
// in sleep mode, IO state is maintained, RAM is retained, and application will resume after waking up
// in deepsleep mode, IOs float, RAM is lost, and application will start from beginning after waking up
// if deep_sleep == true, device will enter deepsleep mode
static bool deep_sleep = false;

mDot* dot = NULL;
lora::ChannelPlan* plan = NULL;

Serial pc(USBTX, USBRX);

void ReadBatteryLevelAndSend()
// get some dummy data and send it to the gateway
{
   std::vector<uint8_t> tx_data;
   AnalogIn adc_batt(A2); 
   float    voltage;
   uint16_t u16_read;
   uint8_t  u8_send;
   u16_read = adc_batt.read_u16();   // read ADC value
   voltage = u16_read / 65535.0 * 3.3 * 2;  //convert to voltage level
   u8_send = voltage * 10;  //use integer and 1 digit of decimal number
   logInfo("Battery Level: Voltage:%f [ADC:%u], Upload Value:%d(0x%x)", voltage, (int)u16_read, (int)u8_send, (int)u8_send);
   tx_data.push_back(u8_send);
   dot->setAppPort(10);   //give a port number to distinglish data source
   send_data(tx_data);
}

uint16_t modbus_crc16(const uint8_t *buf, int length) {
 uint16_t crc = 0xFFFF;
 int i,j;
 uint8_t LSB;
 for (i = 0; i < length; i++) {
  crc ^= buf[i];
  for (j = 0; j < 8; j++) {
   LSB= crc & 1;
   crc = crc >> 1;
   if (LSB) {
    crc ^= 0xA001;
   }
  }
 }
 return crc;
}

#define RS485_BAUDRATE          9600
#define RS485_BITS              8
#define RS485_PARITY            mbed::SerialBase::None
#define RS485_STOPBITS          1
#define RS485_BTYETIME_MS       1
#define RS485_BTYETIME_US       1042
const  int buffer_size = 256;
uint8_t buffer[buffer_size];


void ReadRS485SensorAndSend()
// get some dummy data and send it to the gateway
{
   std::vector<uint8_t> tx_data;

   
   DigitalOut rs485RWctl(PA_11);
   Serial rs485(SERIAL_TX, SERIAL_RX);   
   
   rs485RWctl = 0; //set low to stay in receiver mode
   rs485.baud(RS485_BAUDRATE);
   rs485.format(RS485_BITS, RS485_PARITY , RS485_STOPBITS);   
   //clear modbus buffer
   {
   Timer t;
   t.reset(); t.start();
   int timeup_ms = RS485_BTYETIME_MS * 4;    // let Modbus Frame Time Out
   while (t.read_ms() < timeup_ms)
      {
      if (rs485.readable())
          {
          timeup_ms = RS485_BTYETIME_MS * 3;   // modbus frame timeout
          rs485.getc();
          t.reset();  
          }
      }
   t.stop();   
   }
   
   //sent Modbus command  
   {
   int      trans_count = 0;    
   uint16_t crc16; 
   buffer[0]= 0x0A; //Addr
   buffer[1]= 0x04; //Func
   buffer[2]= 0x00; //Register (High Byte)     
   buffer[3]= 0x01; //Register (Low Byte)      
   buffer[4]= 0x00; //Quantity (High Byte)     
   buffer[5]= 10;   //Quantity (Low Byte)        
   crc16 = modbus_crc16(buffer, 6);
   buffer[6]= (crc16) & 0x00FF;  //CRC16 (Low Byte)
   buffer[7]= (crc16 >> 8) & 0x00FF;  //CRC16 (High Byte)

   rs485RWctl = 1; //set high to swtich to transmitor mode
   while ( trans_count < 8)
    {
    if (rs485.writable())
      {
      rs485.putc(buffer[trans_count]);
      trans_count++;
      wait_ms(RS485_BTYETIME_MS); 
      }
    }
   wait_ms(RS485_BTYETIME_MS * 2);    // let last byte transmit completely
   rs485RWctl = 0; //set low to switch back receiver mode
   logDebug("Send a Modbus command to RS485"); 
   }   
   
   //retrieve Modbus Data 
   {
   Timer t;
   t.reset();  t.start();
   int timeup_ms = 200;    // modbus command timeout, depend on devices
   int hasReceivedBytes = 0;
   int receivedBytes = 0;
        
   while (t.read_ms() < timeup_ms)
      {
      if (rs485.readable())
          {
          timeup_ms = RS485_BTYETIME_MS * 3;   // modbus frame timeout
          t.reset();  
          buffer[receivedBytes] =  rs485.getc();
          receivedBytes++;
          hasReceivedBytes ++;
          }
      }
   t.stop();   
   if ( hasReceivedBytes > 2)
      {
      uint16_t crc16; 
      crc16 = modbus_crc16(buffer, hasReceivedBytes -2 );
      if ( (buffer[hasReceivedBytes - 1] == ((crc16 >> 8) & 0x00FF)) &&
           (buffer[hasReceivedBytes - 2] == ((crc16 ) & 0x00FF)) )
          {
          logDebug("Received Modbus %d bytes correctly", hasReceivedBytes);
          logInfo("Received Modbus Data:%s", mts::Text::bin2hexString(buffer,hasReceivedBytes).c_str());       
          if (( buffer[0] == 0x0A) &&  ( buffer[1]== 0x04) && ( buffer[2]== 20))
            {
            // send data to LoRaWAN gateway
            tx_data.push_back(buffer[3]);  //temperature high byte
            tx_data.push_back(buffer[4]);  //temperature low byte
            tx_data.push_back(buffer[9]);  //Vibration X RMS high byte
            tx_data.push_back(buffer[10]);  //Vibration X RAS low byte
            tx_data.push_back(buffer[11]);  //Vibration Y RMS high byte
            tx_data.push_back(buffer[12]);  //Vibration Y RAS low byte
            tx_data.push_back(buffer[13]);  //Vibration Z RMS high byte
            tx_data.push_back(buffer[14]); //Vibration Z RMS low byte
            tx_data.push_back(buffer[21]); //Power Line Freq high byte
            tx_data.push_back(buffer[22]); //Power Line Freq low byte
            logInfo("Sent 10 bytes Data to LoraWAN Gateway");
            dot->setAppPort(11);   //give a port number to distinglish data source
            send_data(tx_data);
            logDebug("Sent Data to LoraWAN Gateway finished");
            }
          else
            logDebug("Received Modbus Frame is not sensor data");
          }
      else
          {
          logDebug("Received Modbus Frame CRC error");
          logDebug("Received Modbus Data:%s", mts::Text::bin2hexString(buffer,hasReceivedBytes).c_str());               
          }
       }
    else
       {
       if (0 == hasReceivedBytes)
           logDebug("Modbus Command Timeout");
       else    
           logDebug("Received Modbus Frame Error");
       }
   }
}

int main() {
    // Custom event handler for automatically displaying RX data
    RadioEvent events;

    pc.baud(115200);

#if defined(TARGET_XDOT_L151CC)
    i2c.frequency(400000);
#endif

    mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);
    
#if CHANNEL_PLAN == CP_US915
    plan = new lora::ChannelPlan_US915();
#elif CHANNEL_PLAN == CP_AU915
    plan = new lora::ChannelPlan_AU915();
#elif CHANNEL_PLAN == CP_EU868
    plan = new lora::ChannelPlan_EU868();
#elif CHANNEL_PLAN == CP_KR920
    plan = new lora::ChannelPlan_KR920();
#elif CHANNEL_PLAN == CP_AS923
    plan = new lora::ChannelPlan_AS923();
#elif CHANNEL_PLAN == CP_AS923_JAPAN
    plan = new lora::ChannelPlan_AS923_Japan();
#elif CHANNEL_PLAN == CP_IN865
    plan = new lora::ChannelPlan_IN865();
#endif
    assert(plan);

    dot = mDot::getInstance(plan);
    assert(dot);

    // attach the custom events handler
    dot->setEvents(&events);

    if (!dot->getStandbyFlag()) {
        logInfo("mbed-os library version: %d.%d.%d", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);

        // start from a well-known state
        logInfo("defaulting Dot configuration");
        dot->resetConfig();
        dot->resetNetworkSession();

        // make sure library logging is turned on
        dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

        // update configuration if necessary
        if (dot->getJoinMode() != mDot::OTA) {
            logInfo("changing network join mode to OTA");
            if (dot->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
                logError("failed to set network join mode to OTA");
            }
        }
        // in OTA and AUTO_OTA join modes, the credentials can be passed to the library as a name and passphrase or an ID and KEY
        // only one method or the other should be used!
        // network ID = crc64(network name)
        // network KEY = cmac(network passphrase)
        //update_ota_config_name_phrase(network_name, network_passphrase, frequency_sub_band, network_type, ack);
        update_ota_config_id_key(network_id, network_key, frequency_sub_band, network_type, ack);

        // configure network link checks
        // network link checks are a good alternative to requiring the gateway to ACK every packet and should allow a single gateway to handle more Dots
        // check the link every count packets
        // declare the Dot disconnected after threshold failed link checks
        // for count = 3 and threshold = 5, the Dot will ask for a link check response every 5 packets and will consider the connection lost if it fails to receive 3 responses in a row
        update_network_link_check_config(3, 5);

        // enable or disable Adaptive Data Rate
        dot->setAdr(adr);

        // Configure the join delay
        dot->setJoinDelay(join_delay);

        // save changes to configuration
        logInfo("saving configuration");
        if (!dot->saveConfig()) {
            logError("failed to save configuration");
        }

        // display configuration
        display_config();
    } else {
        // restore the saved session if the dot woke from deepsleep mode
        // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
        logInfo("restoring network session from NVM");
        dot->restoreNetworkSession();
    }

    dot->setLogLevel(mts::MTSLog::DEBUG_LEVEL);
    TCA6424_init();

    while (true) {
        LED_D6(true);

        
        LED_D5(true);                
        // join network if not joined
        if (!dot->getNetworkJoinStatus()) {
            join_network();
        }
        LED_D5(false);
        
        LED_D4(true);                
        // Read Sensor Data and Upload it to gateway
//        mts::MTSLog::setLogLevel(mts::MTSLog::DEBUG_LEVEL); // enable debug log
//       ReadBatteryLevelAndSend();
        ReadRS485SensorAndSend();
        LED_D4(false);        

        // if going into deepsleep mode, save the session so we don't need to join again after waking up
        // not necessary if going into sleep mode since RAM is retained
        if (deep_sleep) {
            logInfo("saving network session to NVM");
            dot->saveNetworkSession();
        }
        
        LED_D6(false);

        // ONLY ONE of the three functions below should be uncommented depending on the desired wakeup method
        //sleep_wake_rtc_only(deep_sleep);
        //sleep_wake_interrupt_only(deep_sleep);
        sleep_wake_rtc_or_interrupt(deep_sleep);
    }
 
    return 0;
}

#endif
