#include "mbed.h"
#include "HIDScope.h"
#include "BiQuad.h"
#include "MODSERIAL.h"
#include "FastPWM.h"
#include "QEI.h"

#include <cmath>    // Included to use different math operations

#include "Servo.h"  // Included to control the servo motor 

Servo myservo(D13);         // To control the servo motor

DigitalIn button1(SW2);     // Button to go to the next state or to choose one of two game modes when in state START_GAME
DigitalIn button2(SW3);     // Button to choose one of two game modes when in state START_GAME or to move the gripper in the operating mode

DigitalOut ledr(LED_RED);   // Define the output for the three different LED's on the K64F. These will be used to show in which state the robot currently is in
DigitalOut ledg(LED_GREEN);
DigitalOut ledb(LED_BLUE);

AnalogIn S0(A0);    // Define the different analog inputs for the four EMG shields that are connected to the 
AnalogIn S1(A1);
AnalogIn S2(A2);
AnalogIn S3(A3);

DigitalOut motor1Direction(D7);     // Digital out to control the direction of motor1
FastPWM motor1Velocity(D6);         // FastPWM to control the velocity of motor1
DigitalOut motor2Direction(D4);     // Digital out to control the direction of motor2
FastPWM motor2Velocity(D5);         // FastPWM to control the velocity of motor2

// Encoder for motor1 and motor2
QEI Encoder1(D8,D9,NC,8400);
QEI Encoder2(D11,D10,NC,8400);

Ticker measurecontrol;      // Ticker function for the measurements

Serial pc(USBTX, USBRX);    // Used to get information about the different states and choises if a laptop is attached

double PI = 3.14159265358;  // Defined pi because it is used multiple times
volatile double timeinterval = 0.001f;      // Time interval of the Ticker function
volatile double frequency = 17000.0;        // Motorfrequency of both motors
double period_signal = 1.0/frequency;       // Convert to period of the signal

/*
 Motor calibration is done by manually moving the arms to the start position. This is the position in which the arms
 that are connected to the motors are in line with each other. If they are in the correct position, the robot needs
 to be reset. Then this programm will run and yendeffector and xendeffector are set to be 10.6159 and 0 which are the
 correct physical positions on the board.
*/
double yendeffector = 10.6159;
double xendeffector = 0;

/*
 Boolean that keeps track in which configuration the hook is
 when 0, the hook is ready to grab a ring
 when 1, it is ready to drob an object if it grabbed one
*/
bool pressed = 0;

int previous_state_int; // Keeps track of the previous state to check if the state has changed
double motorvalue1;     // Motorvalue that has to be send to motor1
double motorvalue2;     // Motorvalue that has to be send to motor2

// Define the different states in which the robot can be
enum States {MOTORS_OFF, EMG_CALIBRATION, START_GAME,
             DEMO_MODE, OPERATING_MODE, END_GAME
            };

// Default state is the state in which the motors are turned off
States MyState = MOTORS_OFF;


// Initialise the functions
double PID_controller1(double error1);
double PID_controller2(double error2);
void getmeasuredposition(double & measuredposition1, double & measuredposition2);
void getreferenceposition(double & referenceposition1, double & referenceposition2);
void sendtomotor(double motorvalue1, double motorvalue2);
void measureandcontrol();
void motorsoff();
void measure_data(double &f_y0, double &f_y1, double &f_y2, double &f_y3);
void det_max(double y, double &max_y);
void emgcalibration();
void startgame() ;
void demo_mode();
void operating_mode();
void endgame();
void New_State();



int main()
{
    ledr = 1;   // Set all LEDs to off
    ledg = 1;
    ledb = 1;
    pc.baud(115200);    // Define bitrate for the communication
    pc.printf("Starting...\r\n\r\n");

    motor1Velocity.period(period_signal);   // Set the period of the PWMfunction
    motor2Velocity.period(period_signal);   // Set the period of the PWMfunction

    previous_state_int = (int)MyState;  // Save previous state to compare with current state and possibly execute New_State()
    // in the while loop

    New_State();    // For initialising the state of the robot run New_State() once

    while(true) {
        if ( (previous_state_int - (int)MyState) != 0 ) {
            New_State();    // If current state is not previous state run New_State()
        }
    }
}


// Parallel PID controller to calculate the error for motor1
double PID_controller1(double error1)
{
    static double error_integral1 = 0;  // Keep track of the integral error within this function
    static double error_prev1 = error1; // Safe previous error to use for the derivative part

    // Low-pass filter to reduce noise from the derivative part
    static BiQuad LowPassFilter(0.0640, 0.1279, 0.0640, -1.1683, 0.4241);

    // PID variables: we assume them to be the same for both motors, found by trial and error
    double Kp = 63;
    double Ki = 3.64;
    double Kd = 5;

    //Proportional part:
    double u_k1 = Kp * error1;

    // Integreal part
    error_integral1 = error_integral1 + error1 * timeinterval;
    double u_i1 = Ki*error_integral1;

    // Derivate part
    double error_derivative1 = (error1 - error_prev1)/timeinterval;
    double filtered_error_derivative1 = LowPassFilter.step(error_derivative1);
    double u_d1 = Kd * filtered_error_derivative1;
    error_prev1 = error1;

    //sum all parts and return it
    return u_k1 + u_i1 + u_d1;
}


// Parallel PID controller to calculate the error for motor2
double PID_controller2(double error2)
{
    static double error_integral2 = 0;  // Keep track of the integral error within this function
    static double error_prev2 = error2; // Safe previous error to use for the derivative part

    // Low-pass filter to reduce noise from the derivative part
    static BiQuad LowPassFilter(0.0640, 0.1279, 0.0640, -1.1683, 0.4241);

    // PID variables: we assume them to be the same for both motors, found by trial and error
    double Kp = 63;
    double Ki = 3.64;
    double Kd = 5;

    //Proportional part:
    double u_k2 = Kp * error2;

    // Integreal part
    error_integral2 = error_integral2 + error2 * timeinterval;
    double u_i2 = Ki*error_integral2;

    // Derivate part
    double error_derivative2 = (error2 - error_prev2)/timeinterval;
    double filtered_error_derivative2 = LowPassFilter.step(error_derivative2);
    double u_d2 = Kd * filtered_error_derivative2;
    error_prev2 = error2;

    //sum all parts and return it
    return u_k2 + u_i2 + u_d2;
}


// Calculate the current configuration of both motors
void getmeasuredposition(double & measuredposition1, double & measuredposition2)
{
    // Obtain the counts of motors 1 and 2 from the encoder
    int countsmotor1 = Encoder1.getPulses();
    int countsmotor2 = Encoder2.getPulses();

    // Obtain the measured position for motor 1 and 2
    // Devide by 8400 because 8400 counts is one rotation and times 2 to calculate the measuredposition as a fraction of the rotation
    measuredposition1 = ((double)countsmotor1) / 8400.0f * 2.0f;
    measuredposition2 = ((double)countsmotor2) / 8400.0f * 2.0f;
}


// Calculates the desired configuration of both motors by using the inverse kinematics
void getreferenceposition(double & referenceposition1, double & referenceposition2)
{
    //Measurements of the arm
    double L0=1.95;     // Distance from both motors to (0,0), (0,0) is between motors 1 and 2
    double L1=15;       // Length of the arms attached to motor 1 and 2
    double L2=20;       // Length of the arms attached to the end-effector

    //Inverse kinematics: given the end position, what are the desired motor angles of 1 and 2 (explained in the report)
    double desiredmotorangle1, desiredmotorangle2;
    desiredmotorangle1 = (atan2(yendeffector,(L0+xendeffector))*180.0/PI + acos((pow(L1,2)+pow(L0+xendeffector,2)+pow(yendeffector,2)-pow(L2,2))/(2*L1*sqrt(pow(L0+xendeffector,2)+pow(yendeffector,2))))*180.0/PI)-180.0;
    desiredmotorangle2 = (atan2(yendeffector,(L0-xendeffector))*180.0/PI + acos((pow(L1,2)+pow(L0-xendeffector,2)+pow(yendeffector,2)-pow(L2,2))/(2*L1*sqrt(pow(L0-xendeffector,2)+pow(yendeffector,2))))*180.0/PI)-180.0;

    //Convert motor angles to counts
    double desiredmotorrounds1, desiredmotorrounds2;
    desiredmotorrounds1 = (desiredmotorangle1)/360.0;
    desiredmotorrounds2 = (desiredmotorangle2)/360.0;

    // Assign this to new variables because otherwise it doesn't work
    referenceposition1 = desiredmotorrounds1;
    referenceposition2 = desiredmotorrounds2;
}


// Send the absolute motorvalues as velocity to the motors and set the motor directions by checking if the motorvalues are positive
void sendtomotor(double motorvalue1, double motorvalue2)
{
    // Define the absolute motor values
    double absolutemotorvalue1, absolutemotorvalue2;
    absolutemotorvalue1 = fabs(motorvalue1);
    absolutemotorvalue2 = fabs(motorvalue2);

    // If absolutemotorvalueocity is greater than 1, reduce to 1, otherwise remain absolutemotorvalue
    absolutemotorvalue1 = absolutemotorvalue1 > 1.0f ? 1.0f : absolutemotorvalue1;
    absolutemotorvalue2 = absolutemotorvalue2 > 1.0f ? 1.0f : absolutemotorvalue2;

    // Send the absolutemotorvalue to the motors
    motor1Velocity = absolutemotorvalue1;
    motor2Velocity = absolutemotorvalue2;

    // Determine the motor direction. Boolean output: true gives counterclockwise direction, false gives clockwise direction
    motor1Direction = (motorvalue1 > 0.0f);
    motor2Direction = (motorvalue2 > 0.0f);
}


// function to call different functions and control the motors
void measureandcontrol()
{
    // Get the reference positions of motor 1 and 2
    double reference1, reference2;
    getreferenceposition(reference1, reference2);

    // Get the measured positions of motor 1 and 2
    double measured1, measured2;
    getmeasuredposition(measured1, measured2);

    // Calculate the motor values
    motorvalue1 = PID_controller1(reference1 - measured1);
    motorvalue2 = PID_controller2(reference2 - measured2);
    sendtomotor(motorvalue1, motorvalue2);
}


// Function that is called once in the MOTORS_OFF state, motorvelocities are zero, will run the while loop until button1 is pressed.
void motorsoff() {
    sendtomotor(0.0,0.0);           // Set motor velocity to 0

    bool whileloop_boolean = true;  // Boolean for the while loop

    while (whileloop_boolean) {
        if (button1.read() == 0) {                  // If button1 is pressed:
            MyState = (States)((int)MyState+1);     // set MyState to EMG_CALIBRATION and exit the while loop
            whileloop_boolean = false;              // by making whileloop_boolean equal to false
            wait(0.5f);
        }
    }
}

/*
 Uses a high-pass filter, rectifier and a low-pass filter on all EMG signals and then normalises the signal
 by deviding by the maximum value that was measured during the EMG_CALIBRATION state. This maximum value
 is determined within this function if the state is EMG_CALIBRATION
*/
void measure_data(double &f_y0, double &f_y1, double &f_y2, double &f_y3)
{
    // High-pass
    double hb0 = 0.9169;  // Coefficients from the following formula:
    double hb1 = -1.8338; //
    double hb2 = 0.9169;  //        b0 + b1 z^-1 + b2 z^-2
    double ha0 = 1.0;     // H(z) = ----------------------
    double ha1 = -1.8268; //        a0 + a1 z^-1 + a2 z^-2
    double ha2 = 0.8407;  //

    // Low-pass
    double lb0 = 0.000083621;  // Coefficients from the following formula:
    double lb1 = 0.0006724; //
    double lb2 = 0.000083621;  //        b0 + b1 z^-1 + b2 z^-2
    double la0 = 1.0;     // H(z) = ----------------------
    double la1 = -1.9740; //        a0 + a1 z^-1 + a2 z^-2
    double la2 = 0.9743;  //

    static double max_y0 = 0.001;   // Maxima have to be static because they have to be changed during the
    static double max_y1 = 0.001;   // calibration and have to be used afterwards to normalise the signal
    static double max_y2 = 0.001;
    static double max_y3 = 0.001;

    static BiQuad hFilter0(hb0,hb1,hb2,ha0,ha1,ha2);  // Create 4 equal high-pass filters used for the different EMG signals
    static BiQuad hFilter1(hb0,hb1,hb2,ha0,ha1,ha2);
    static BiQuad hFilter2(hb0,hb1,hb2,ha0,ha1,ha2);
    static BiQuad hFilter3(hb0,hb1,hb2,ha0,ha1,ha2);

    static BiQuad lFilter0(lb0,lb1,lb2,la0,la1,la2);  // Create 4 equal low-pass filters used for the different EMG signals
    static BiQuad lFilter1(lb0,lb1,lb2,la0,la1,la2);
    static BiQuad lFilter2(lb0,lb1,lb2,la0,la1,la2);
    static BiQuad lFilter3(lb0,lb1,lb2,la0,la1,la2);

    f_y0 = hFilter0.step(S0);     // Apply high-pass filters on the different EMG signals
    f_y1 = hFilter1.step(S1);
    f_y2 = hFilter2.step(S2);
    f_y3 = hFilter3.step(S3);

    f_y0 = abs(f_y0);   // Apply rectifier on the different EMG signals
    f_y1 = abs(f_y1);
    f_y2 = abs(f_y2);
    f_y3 = abs(f_y3);

    f_y0 = lFilter0.step(f_y0); // Apply low-pass filters on the different EMG signals
    f_y1 = lFilter1.step(f_y1);
    f_y2 = lFilter2.step(f_y2);
    f_y3 = lFilter3.step(f_y3);


    if (MyState == EMG_CALIBRATION) {
        det_max(f_y0, max_y0);      // Determine the maximum value of the EMG signals during the EMG_CALIBRATION state
        det_max(f_y1, max_y1);      // If a new maximum is found, the max_y_ value is adjusted
        det_max(f_y2, max_y2);
        det_max(f_y3, max_y3);

    } else if ((int)MyState > 3) {
        f_y0 = f_y0/max_y0;     // Normalise the RMS value by dividing by the maximum RMS value
        f_y1 = f_y1/max_y1;     // This is done during the states with a value higher than 3, as this is when you start the operating mode
        f_y2 = f_y2/max_y2;
        f_y3 = f_y3/max_y3;
    }

}


// Function to determine if y is larger than max_y
void det_max(double y, double &max_y)
{
    max_y = max_y < y ? y : max_y; // if max_y is smaller than y, set y as new max_y, otherwise keep max_y
}


// Function that is attached to the measurecontrol ticker when the state is EMG_CALIBRATION, the ticker will run this function for a specific amount of time
void emgcalibration()
{
    double y0, y1, y2, y3;           // variables to safe the output from measure_data

    measure_data(y0, y1, y2, y3);   // Calculate y values

    double duration = 20.0;                 // Duration that the emgcalibration() has to run
    int rounds = (duration / timeinterval); // Determine the amount of times this function has to run to run for the duration time
                                            // Rounds is an integer so the value of duration / timeinterval is floored

    static int counter = 0;     // Counter which keeps track of the amount of times the function has executed
    if (counter >= rounds) {
        MyState = START_GAME;       // If counter is larger than rounds, change MyState to the START_GAME state
        ledb = 1;                   // Turn the blue LED off
        measurecontrol.detach();    // detach emgcalibration() from the ticker function
    } else {
        counter++;  // Else increase counter by 1
    }
    
    int rounds_led = 0.1f / timeinterval;  // Amount of rounds that the blue LED has to be on and off
    if (counter % rounds_led == 0) {
        ledb = !ledb;
    }
}


// Function that is executed when the state is changed to START_GAME, will run a while loop until
// button1 or 2 are pressed to choose the game mode
void startgame()
{
    // Print messages with information for the user
    pc.printf("Please choose which game you would like to start:\r\n");

    pc.printf("- Press button1 to start the demo mode\r\n     Demo mode is a mode in which the different movements of the robot are shown.\r\n");
    pc.printf("     It will move in straight lines to show that the robot meets the requirements.\r\n");
    pc.printf("     It will also show how it is able to grab and lift objects which are on the board.\r\n\r\n");

    pc.printf("- Press button2 to start the operating mode\r\n     The operating mode is a mode in which you can control the robot by flexing the muscles to which the electrodes are attached.\r\n");
    pc.printf("     You will be able to move the arm and use the gripper to try and grab and lift objects from the board to the container.\r\n\r\n");

    while (MyState == START_GAME) {
        if (button1.read() == 0) {
            MyState = DEMO_MODE;        // Press button1 to choose DEMO_MODE
            wait(0.5f);
        } else if (button2.read() == 0) {
            MyState = OPERATING_MODE;   // Press button2 to choose OPERATING_MODE
            wait(0.5f);
        }
    }
}


// Function that is executed when the state is changed to DEMO_MODE, will move the end-effector
// to 5 different positions to show it can move in straight lines and then go back to the START_GAME state
void demo_mode()
{
    // CURRENTLY STILL NOT WORKING, ARMS WILL MOVE TO SINGULARITY, WATCH OUT WITH THIS MODE
    
    // 5 pre determined positions of the end-effector to show it can move in straight lines
    xendeffector=-5;
    yendeffector=10.6159;
    wait(0.5);

    xendeffector=10;
    yendeffector=25.6159;
    wait(0.5);

    xendeffector=-14;
    yendeffector=21.6159;
    wait(0.5);

    xendeffector=10;
    yendeffector=11.6159;
    wait(0.5);

    // Last position is the start position
    xendeffector=0;
    yendeffector=10.6159;
    wait(0.5);

    MyState = START_GAME;     // Go back to START_GAME mode
}

/*
 Function that is attached to the measurecontrol ticker when the state is OPERATING_MODE
 Turns the green LED on and off, calls measuredata(), changes the configuration of the hook
 when button2 is pressed, sets a new position for the end-effector and goes to the state
 END_GAME when button1 is pressed.
*/
void operating_mode()
{
    // Turn green LED on or off
    static int counter = 0;     // Keep track of the amount of times the function has executed
    int rounds_led = 0.1 / timeinterval;    // Amount of rounds that the green LED has to be on and off
    if (counter % rounds_led == 0) {
        ledg = !ledg;
        counter = 0;
    }
    counter++;
    
    double y0,y1,y2,y3;
    measure_data(y0,y1,y2,y3);  // Calculate y values
    
    //servo besturing
    if(button2.read() == 0 && pressed == 0) {
        // if button2 is pressed, set y0, y1, y2 and y3 to zero, change pressed and change
        // the servo configuration one step at a time to reduce the speed of the rotation
        y0 = 0;
        y1 = 0;
        y2 = 0;
        y3 = 0;
        pressed = 1;
        for(int i=0; i<100; i++) {
            myservo = i/100.0;
        }
    }
    else if(button2.read() == 0 && pressed == 1) {
        // if button2 is pressed, set y0, y1, y2 and y3 to zero, change pressed and change
        // the servo configuration one step at a time to reduce the speed of the rotation
        y0 = 0;
        y1 = 0;
        y2 = 0;
        y3 = 0;
        pressed = 0;
        for(int i=100; i>0; i--) {
            myservo = i/100.0;
        }
    }
    
    // The threshold has been set to 0.4 by trial and error
    double threshold = 0.4;     // When the analysed signal is above this threshold, the position of the end-effector is changed
    double dr = 0.01;           // The change in position with each execution
    
    if(y0 > threshold && xendeffector < 16) {
        xendeffector=xendeffector+dr;
    }
    else if(y1 > threshold && xendeffector > -16) {
        xendeffector=xendeffector-dr;
    }
    // The threshold was adjusted for the y-position because this worked better, this maybe had to be done because of a broken EMG cable
    if(y2 > threshold+0.2 && yendeffector < 28) {
        yendeffector=yendeffector+dr;
    }
    else if(y3 > threshold+0.2 && yendeffector > 8) {
        yendeffector=yendeffector-dr;
    }

    // When button1 is pressed, run while loop until button is released again to prevent bouncing
    if (button1.read() == 0) {
        bool buttonss = true;
        while (buttonss) {
            if (button1.read() == 1) {
                buttonss = false;
            }
        }
        pc.printf("The game has ended, will move the end-effector to (0,0), put the motors off and will return to the state START_GAME");
        MyState = END_GAME;         // Set MyState to END_GAME
        measurecontrol.detach();    // Detach the operating_mode() function from measurecontrol
    }

    measureandcontrol();    // Call measureandcontrol() to adjust the motors
}


// Function that is executed once when state is changed to END_GAME, returns the end-effector to
// the start position and detaches the measureandcontrol function from the measurecontrol ticker
void endgame()
{
    measurecontrol.attach(measureandcontrol,timeinterval);
    
    double tempx = xendeffector;
    double tempy = yendeffector-10.6159;
    double steps = 20;  // Change the position of the end-effector in a certain amount of steps to the start-position
    // This is done in steps to reduce abrupt movement of the end-effector
    
    for (int i = 0; i < steps; i++) {
        xendeffector = xendeffector - tempx/steps;
        yendeffector = yendeffector - tempy/steps;
        wait(0.1);
    }
    measurecontrol.detach();    
    MyState = START_GAME;       // Return the state to START_GAME so a new game mode can be chosen again
}


// Function that is called by the while loop in the main function if the current state is changed
// Attaches or executes the functions belonging to the current state
void New_State()
{
    previous_state_int = (int)MyState;      // Change previous state to current state

    switch (MyState) {
        case MOTORS_OFF :
            pc.printf("\r\nState: Motors turned off\r\n");
            ledb = 0;       // blue led turns on when you enter MOTORS_OFF state
            motorsoff();
            ledb = 1;       // blue led turns off when you exit MOTORS_OFF state
            break;

        case EMG_CALIBRATION :
            pc.printf("\r\nState: EMG Calibration\r\n");
            pc.printf("Emg calibration mode will run for 20 seconds. Try to flex the muscles to which the electrodes are attached as hard as possible during this time.\r\n");
            measurecontrol.attach(emgcalibration,timeinterval); // Attach emgcalibration() to the measurecontrol ticker
            break;

        case START_GAME :
            pc.printf("\r\nState: Start game\r\n");
            ledr = 0;       // red led is on when you enter START_GAME state
            ledb = 0;       // blue led is on when you enter START_GAME state
            ledg = 0;       // green led is on when you enter START_GAME state
            startgame();
            break;

        case DEMO_MODE :
            pc.printf("\r\nState: Demo mode\r\n");
            measurecontrol.attach(measureandcontrol,timeinterval); // Attach measureandcontrol() to the measurecontrol ticker
            ledr = 1;       // red led is on when you enter START_GAME state
            ledb = 0;       // blue led is on when you enter START_GAME state
            ledg = 0;       // green led is on when you enter START_GAME state
            demo_mode();
            measurecontrol.detach();    // Detach measureandcontrol() grom the measurecontrol ticker
            break;

        case OPERATING_MODE :
            pc.printf("\r\nState: Operating mode\r\n");
            ledr = 1;       // red led is off when you enter OPERATING_MODE state
            ledb = 1;       // blue led is off when you enter OPERATING_MODE state
            ledg = 0;       // green led is on when you enter OPERATING_MODE state
            measurecontrol.attach(operating_mode,timeinterval); // Attach operating_mode() to the measurecontrol ticker
            break;

        case END_GAME :
            pc.printf("\r\nState: End of the game\r\n");
            ledr = 0;       // red led is on when you enter OPERATING_MODE state
            ledb = 1;       // blue led is off when you enter OPERATING_MODE state
            ledg = 1;       // green led is off when you enter OPERATING_MODE state
            endgame();
            break;

        default :
            pc.printf("\r\nSTATE IS CHANGED TO A NONEXISTING STATE!\r\n");
            pc.printf("\r\nMotors are turned off\r\n");
            sendtomotor(0.0,0.0);
            ledg = 1;
            ledb = 1;
            while (true) {
                ledr = !ledr;
                wait(0.1f);
            }
    }
}
