/*
 * tlc5940 - Interface to operate TI's IC TLC5940
 * Copyright (C) 2010 German Bassi.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef MBED_TLC5940_H
#define MBED_TLC5940_H

#include "mbed.h"

/** TI's TLC5940 interface class
 *
 * Example:
 * @code
 * // Turn on all the outputs
 * # include "mbed.h"
 * # include "tlc5940.h"
 * 
 * int main() {
 *     int DC_data[2*96];
 *     int GS_data[2*16];
 *
 *     // Dot Correction Values
 *     for (int i=0; i<2*96; i++) DC_data[i] = 1;
 *     // Grayscale Values
 *     for (int i=0; i<2*16; i++) GS_data[i] = 0xFF;
 *
 *     // Create object
 *     tlc5940 tlc_driver(2, DC_data);
 *     // Send data
 *     tlc_driver.send_data(GS_data);
 *
 *     while (1) {}
 * }
 * @endcode
 */
class tlc5940 {
private:
    bool first_cycle_flag;
    int GSCLK_counter, data_counter;
    int aux_value, aux_ind;
    int num_ic;

    DigitalOut VPROG;
    DigitalOut SIN;
    DigitalOut SCLK;
    DigitalOut XLAT;
    DigitalOut BLANK;
    DigitalOut DCPROG;
    DigitalOut GSCLK;

    DigitalIn SOUT;
    DigitalIn XERR;

public:
    /** Create a tlc5940 interface object connected to some specifics pins
     *
     * @param num_ics Number of TLC5940 connected in series
     * @param DC_data[] Dot Correction values for initialization
     */
    tlc5940 (int num_ics, int DC_data[]);

    /** Send the specified set of grayscale values
     *
     * @param data[] Array of 12-bit Grayscale values for transmission
     */
    void send_data (int data[]);
};

#endif