#ifndef _HIDSCOPE_H_
#define _HIDSCOPE_H_

#include "mbed.h"
#include "USBHID.h"

/** A simple HID (Human Interface Device) scope
 * - Up to 6 channels of float data is transmitted in a single HID message (64 byte)
 * - Theoretical maximum samplerate of 1kHz (due to HID specifications)
 * - Data can be parsed using a client-side server
 *
 * See the following repository for PC software: https://bitbucket.org/tomlankhorst/hidscope
 * 
 * Example:
 * @code
 * #include "mbed.h"
 * #include "HIDScope.h"        // Require the HIDScope library
 *
 * HIDScope    scope(4);        // Instantize a 2-channel HIDScope object
 * Ticker      scopeTimer;      // Instantize the timer for sending data to the PC 
 * 
 * AnalogIn    a0(A0);          // Using an analog input to obtain data 
 * 
 * int main()
 * {
 *   
 *   // Attach the HIDScope::send function to the timer at a 10.000 us interval (100 Hz)
 *   scopeTimer.attach_us(&scope, &HIDScope::send, 1e4);
 *   
 *   // Read from the analog input in an endless loop. Two channels are written each iteration. 
 *   // Note that the control loop can run at a different frequency (1 kHz in this case)
 *   while(1){
 *       scope.set(0, a0.read());
 *       scope.set(1, a0.read());
 *       wait_us(1000);
 *   };    
 *   
 * }
 * @endcode
*/
class HIDScope {
    public: 
        ///Instantiate the HID Scope
        HIDScope(int channels);
        
        /** Sets the current channel value
          @param ch : integer channel no (0-6)
          @param val : float value
          @return void
        */
        void set(int ch, float val);
        
        /** Sets the current channel value
          @param ch : integer channel no (0-6)
          @param val : integer value
          @return void
        */
        void set(int ch, int val);
        
        /** Sets the current channel value
          @param ch : integer channel no (0-6)
          @param val : boolean value
          @return void
        */
        void set(int ch, bool val);
        
        /** Sets the current channel value
          @param ch : double channel no (0-6)
          @param val : float value
          @return void
        */
        void set(int ch, double val);
        
        /** Sends the channel data to the HID client
          @return void
        */
        void send();
    private:
        USBHID hid;
        HID_REPORT scopeData;
        float* bufferData;
        int channelCount;
};

#endif