#include "CircularArray.h"
#include <stdio.h>
#include <ctype.h>

/**
    @file :     CircularArray.cpp
    @authors :   Radu Marcu, Jacob Williams, Niall Francis, Arron Burch
    
    @section DESCRIPTION
    
    This is the circular array class responsible for managing data stored in the circular array.
    It handles all functions related to the stored measures, including pushing measures to the next available
    space, pulling the first measure, as well as reading and deleting measures. As this is a circular buffer,
    if there is a new measure added and not enough room for it, the oldest measures will be removed in order
    to make space.
    This class was written to be fairly robust; if it is asked to delete or read more measures than it
    contains, the class can adapt and delete or read the highest possible value.
    
*/

CircularArray::CircularArray(int limit, MessageLogger *newLogger)
{
    maxSize = limit;
    array = new Measure[maxSize];
    firstValue = 0;
    currentSize = 0;
    logger = newLogger;
}        

/**
    Adds a new Measure object to the next available space in the buffer.
    
    @param _measure :   Measure object to be added to buffer.
*/
void CircularArray::pushValue(Measure _measure)
{
    if(currentSize >= maxSize)
    {
        firstValue++;
        currentSize--;
    }
    int next = nextSpace();
    array[next] = _measure;
    currentSize++;
}

/**
    Returns the first element in the buffer.
    
    @return array :   First Measure object in the buffer.
*/
Measure CircularArray::pullValue()
{
    return array[firstValue];
}

char temp2[256];

/**
    Prints the newest n measurements from the buffer, using the MessageLogger.
    
    @param n :  The number of measurements to be printed.
*/
void CircularArray::readN(int n)
{
    if(n > currentSize)
    {
        n = currentSize;
        snprintf(temp2, 256, "Not enough measurements taken. Printing all %i measurement(s): \r\n", n);
    }
    else
    {
        snprintf(temp2, 256, "Printing %i measurement(s): \r\n", n);            
    }
                        
    logger->SendMessage(temp2);
        
    int currentElement = nextSpace() - n;
    
    if(currentElement < 0)
    {
        currentElement += maxSize;
    }
    
    char* ptr;
    
    // Iterates through newest n measurements sending the data to be printed via the MessageLogger.
    for(int i = 0; i < n; i++)
    {
        ptr = array[currentElement].date.ToString();
        snprintf(temp2, 256, "%i. %s, T: %f, H: %f, P: %f\n\r",i + 1,ptr , array[currentElement].temperature, array[currentElement].humidity, array[currentElement].pressure);
        logger->SendMessage(temp2);
        currentElement++;
        currentElement = currentElement % maxSize; 
    }
}

/**
    Calls readN to print all measurements that are held in the buffer so far.
*/
void CircularArray::readAll()
{
    logger->SendMessage("Printing all measurementss performed so far: \r\n"); 
    
    readN(currentSize);
}

/**
    Moves firstValue pointer n elements forward to allow old elements to be overwritten.
    
    @param n :  Number of elements to be removed.
*/
void CircularArray::deleteN(int n)
{
    if(n >= maxSize)
    {
        deleteAll();
    }
    else
    {
        firstValue += n;
        currentSize -= n;
        firstValue = firstValue % maxSize;
    }
}

/**
    Resets currently used size of the buffer to 0, allowing for new measurements to overwrite old measurements.
*/
void CircularArray::deleteAll()
{
    snprintf(temp2, 256, "Deleted %i records.", currentSize);
    logger->SendMessage(temp2);
    currentSize = 0;
    
}

/**
    @return currentSize:   The current number of elements used in the buffer.
*/
int CircularArray::getSize()
{
    return currentSize;
}

/**
    Calculates the next available space in the buffer.
    
    @return nextSpace :   The index of the next available space.
*/
int CircularArray::nextSpace()
{
    int nextSpace = firstValue + currentSize;
    nextSpace = nextSpace % maxSize; 
    return nextSpace;   
}