#include "MessageLogger.h"
#include "Message.h"

#define SIGNAL_printMessage 2

/**
    @file :     MessageLogger.cpp
    @authors :   Radu Marcu, Jacob Williams, Niall Francis, Arron Burch
    
    @section DESCRIPTION
    
    This is the MessageLogger class, which is responsible for handling the printing
    of messages. All messages are placed in a mailbox to be printed out in order of
    oldest to newest, with the exception of error messages, which are prioritized by
    the logging thread (main.cpp).
    */


Mail<Message, 16> message_mail;

MessageLogger::MessageLogger()
{
    hasError = false;
    messageCount = 0;
}

/**
    @param logger :     Pointer to the loggingThread.
*/
void MessageLogger::SetThread(Thread* logger)
{
    loggingThread = logger;           
}

/**
    Sends a signal to the loggingThread, indicating an error needs to be printed.
    
    @param errorMessage : Error Message to be sent through the loggingThread.
*/
void MessageLogger::SendError(string errorMessage)
{
    fatalError << errorMessage <<"Terminating Program...\r\n";
    
    // Tells the loggingThread there is an error queued to be printed.
    loggingThread->signal_set(SIGNAL_printMessage);
    hasError = true;
}

Message *newMsg;

/**
    Sends a signal to the loggingThread, indicating a message needs to be printed.
    
    @param message :    Message to be sent through the loggingThread.
*/
void MessageLogger::SendMessage(char* message)
{
    messageLock.lock();
    newMsg = message_mail.alloc();           
    
    // Checks if space in mailbox has been unsuccessfully allocated for the message.
    if (newMsg == NULL) 
    {
        // Sends an error though the loggingThread.
        SendError("ERROR: Message queue is full.\r\n");
        messageLock.unlock();
        return;   
    }

    newMsg->copy(message);

    stat = message_mail.put(newMsg);

    // Checks if message has been unsuccessfully put into the mailbox.
    if (stat == osErrorResource) 
    {
        message_mail.free(newMsg);
        ostringstream error;
        error << "ERROR CODE: " << stat << ", Failed to put message into queue\r\n";
        SendError(error.str());
        messageLock.unlock();
        return;
    }

    messageCount++;
    
    // Tells the logging thread there is a message queued to be printed.
    loggingThread->signal_set(SIGNAL_printMessage);

    printf("\033[1A\n");
    messageLock.unlock();
}

/**
    Used by the loggingThread to initialise the printing of a queued error message.
    
    @return : Whether there is an error ready to be printed.
*/
bool MessageLogger::GetError()
{
    if(hasError)
    {
        PrintError();
        return true;
    }
    
    return false;
}

/**
    Used by the loggingThread to initialise the printing of a queued message.
    
    @return : Whether there is a message ready to be printed.
*/
bool MessageLogger::GetMessage()
{
    if(messageCount > 0)
    {
        PrintMessage();
        return true;
    }
    return false;  
}

/**
    Prints the queued error message to the terminal.
*/
void MessageLogger::PrintError()
{
    printf("%s", fatalError.str().c_str());
    hasError = false;
}

Message *message; 
char copyOfString[256];

/**
    Prints the oldest queued message to the terminal.
*/
void MessageLogger::PrintMessage()
{
    messageLock.lock();
    osEvent evt = message_mail.get();

    // Checks if the message has been retreived from the mailbox successfully.
    if (evt.status == osEventMail) 
    {
        // Copies the content from the mailbox into an array and prints to the terminal.
        message = (Message*)evt.value.p;  
        strncpy(copyOfString, message->getText(), 256);
        message_mail.free(message);
        printf("%s\033[1A\n", copyOfString);               
        
        messageCount--;
    } 
    else // If unsuccessful, send an error to the loggingThread to be printed.
    {
        ostringstream error;
        error << "ERROR CODE: " << evt.status << ", Failed to retrieve message from queue\r\n";
        SendError(error.str());
    }  

    messageLock.unlock();
}