/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Implementantion of a weather station using the BMP180 sensor, the N5110 LCD screen and the mbed NXP LPC1768  microcontroller
@brief to do that the libraries developed by Dr. Craig Evans are going to be used to control the BMP180 sensor and the N5110 LCD screen
@brief a power saving feature was enabled by user the powerControl library by Michael Wei
@brief Revision 1.5.
@author Felipe F. Lopes
@date   May 2015
*/

#ifndef MAIN_H
#define MAIN_H


#include "mbed.h"
#include "N5110.h"
#include "BMP180.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

//Here the initial values for the data are stored, maybe it can be possible to use them though the entire program
//The temperatures are stored in Celsius
#define maxThresTemp 50
#define minThresTemp 0

//Constants that will be used to define states
#define changeMaxTemp 10
#define changeMinTemp 3
#define dontChangeTemp 20
#define mediumGood 5
#define highGood 6
#define mediumBad 7
#define highBad 8
#define notDefined 9

//Pins for the Leds
#define maxTempAlertPin p23
#define minTempAlertPin p22

//Pin values for the SDA and SCL lines of the BMP180 sensor. The data and clock lines respectivly
#define sdaPin p9
#define sclPin p10

//Pins to connect the LCD Screen
#define pwrPin p5
#define scePin p6
#define rstPin p7
#define dcPin p8
#define mosiPin p11
#define sclkPin p13
#define ledPin p24

//Pin to turn the LCD on/off
#define LCDStatePin p18

//pin to turn loggin on/off
#define dataLoggingPin p29

//pin to read from the potentiometer
#define potentiometerPin p20

//pin for button to change Threshold
#define trhesPinButton p25

//pin to buzzer
#define buzzerPin p21

//pin for button for change of units shown
#define  changeUnitsPin p26

/**
@brief Variable to store the value of the forecasted weather
*/
int chanceOfWeather = notDefined;

/**
@brief Counter for the amount of times that the measurement was get on the minute
*/
int minuteCounter = 0;
/**
@brief Counter for the amount of times that the measurement was taken on the hour
*/
int hourCounter = 0;
/**
@brief Counter for the hours
*/
int hoursCounter = 0;

/**
@brief values for initial the threshold temperatures
*/
float maxTemp = 30;
float minTemp = 10;

/**
@brief This variable will track with the embed should show the temperature in celcius or farenheight and change the pressure as well
*/
bool changeUnits=false;

/**
@brief Variable to see if the lcd is on or not
*/
bool isOn= false;

/**
@brief Variable to track if the temperature threshold is going to be changed
*/

int option = dontChangeTemp;

/**
@brief Struct variable that will store the value of the temperature and pressure, initiali the temperature is in celcis and the pressure in mbar
*/

Measurement measurement;

/**
@brief This vector we will store the measurements of pressure over a minute
*/
Measurement measurementMinute[60];

/**
@brief This vector we will store the measurements of pressure over a hour
*/
Measurement measurementHour[60];

/**
@brief This vector we will store the measurements of pressure over four hours
*/
Measurement measurementHours[4];

/**
@brief Pins to conect the leds for the visual warmings
*/
DigitalOut maxTempAlert (maxTempAlertPin);
DigitalOut minTempAlert (minTempAlertPin);

/**
@brief Buzzer to be fired when the temperature is not in the range defined by the thresholds
*/
PwmOut buzzer(buzzerPin);

/**
@brief Creating of the LCD Object
*/
N5110 lcd(pwrPin, scePin, rstPin, dcPin, mosiPin, sclkPin, ledPin);

/**
@brief main part of the project, the sensor
*/
BMP180 sensor (sdaPin,sclPin);

/**
@brief Button to change units
*/
InterruptIn unitsButton(changeUnitsPin);

/**
@brief Button for Trheshold change
*/
DigitalIn thresholdButton (trhesPinButton);

/**
@brief Button to turn LCD on/off
*/
DigitalIn LCDState(LCDStatePin);

/**
@brief Button to turn logging on/off
*/
DigitalIn logging(dataLoggingPin);

/**
@brief Potentiometer read
*/
AnalogIn pot (potentiometerPin);

/**
@brief Local file system
*/
LocalFileSystem local("file");

/** 
@brief serial interface
*/
Serial serial(USBTX,USBRX);

/** 
@brief Ticker to get new temperature and pressure readings
*/
Ticker readUpdates;

/**
@brief Ticker to update the IO
*/
Ticker iO;

/**
@brief Interupt to enable the change on the values of the threshold
*/
InterruptIn enableChange(trhesPinButton);

/**
@brief String to be shown on the LCD
*/
char buffer[14];

/**
@brief contains the time string
*/
char rxString[14];
char date[14];

/**
@brief function to update and check the IO
*/
void ioUpdate();
/**
@brief function to change the date received to a format easily for a human to read
*/
void formatData();

/**
@brief Function to be called to received the UNIX time when sent over the USB
*/
void getTimeUpdate();

/**
@brief This function gets the UNIX time received over the USB and store as an integer on the RAM memory
*/
void setTime();

/**
@brief Function to save power on the mbed by disabling the ethernet port
*/
void powerSaving();

/**
@brief this function reads the temperature and pressure value detected on the BMP180 sensor and store in the RAM memory
*/
void getMeasurements();

/**
@brief This function gets the temperature in celcius, pressure in mbar and the time formated and store in a .csv file
*/
void writeOnFile();

/**
@brief Function to check if the temperature is within the threshold defined, if not it call another function to alert the user.
*/
void checkValues();

/**
@brief Function to be called when the temperature is over the maximun defined threshold
*/
void highTempAction();

/**
@brief Function to be called in case the temperature is lower than the minimum defined threshold
*/
void lowTempAction();

/**
@brief If the temperature is within the define threshold this function is called
*/
void normalTempAction();

/**
@brief After the user decided to change the temperature threshold this function will see which of the threshold need to be changed and read the potentiomenter input
@param option - option to which threshold is going to be changed
*/
void changeThreshold(int op);

/**
@brief Change the units shown in the LCD, from celcius to farenheight and from mbar to atm
*/
void unitsChanger();

/**
@brief function to start the change of the threshold
*/
void enableThresChange();

/**
@brief this function will store the measurements taken every second up to 4 hours, take the average of the readings and call the function to make the weather predictions
*/ 
void storeMeasurement();

/**
@brief based on the values stored and their average this function will make a basic weather prediction 
*/
void makePrediction();

#endif
