#ifndef __SERVICE_HPP__
#define __SERVICE_HPP__

#include "Context.h"

#ifndef TEST_ENVIRONMENT
#include "mbed.h"
#endif

#ifdef CPLUSPLUS_99
#include <stdint.h>
#endif

#include <iostream>
#include <sstream>
#include <string>

#include "Value.hpp"


namespace misnet {
  class Service;
}


class misnet::Service {

public:

  typedef uint8_t MISNET_CODE ;
  typedef uint8_t DEVICE_TYPE ;    //  FC to compile

  enum SERVICE_TYPE {
    SENSOR    = 1, 
    ACTUATOR  = 2, 
    RECORDER   = 3
  } ;

  // A compléter au fur et a mesure et remonter l'info sur les centres applicatifs
  enum SERVICE_ID {
    NOT_IDENTIFIED  = 0 ,   //  Configuration nouvelle ou non référencée
    IKS01A2         = 1 ,   //  Liste des composants
    SMART_TERMINAL  = 2     //  BME280 + .... 
  } ;

  typedef uint8_t  GROUP ;

  //typedef uint8_t VALUE_TYPE ;

  enum STATE {
    ENABLED   = 1,  //   ACTIVE
    DISABLED  = 0   //   ASLEEP
  } ;

  enum ACCESS_TYPE {
    GPIO_    = 1,
    I2C_     = 2,
    SPI_     = 3,
    UART_    = 4
  } ;
 
  typedef uint8_t ACCESS_PIN ;

  enum UP_MODE {
    BY_CHANGE = 0,  // Check that current value differs from previous value
    BY_DELTA  = 1,  // Check a delta between current value and previous value
    BY_RANGE  = 2   // Check whether current value lies within a range
  } ;

  enum REQUEST_MODE {
    IRQ_   = 1, // wakeup by irq
    TIME_  = 2  // wakeup by watchdog timer
  } ; 

  //uint32_t TIMER_DIVIDER ;

  /*
  typedef float DELTA_THRESHOLD ;
  typedef float LOW_THRESHOLD_LIMIT ;
  typedef float HIGH_THRESHOLD_LIMIT ;
  */

  enum ACTION {
    MESSAGE         = 1, // Send Message
    MESSAGERELAY    = 2  // Send Message + ON/OFF internal relay
  } ;

  enum OUTPUT_MODE {
    IO        = 1, // ON/OFF 
    PWD       = 2  // PWD modulation 
  } ;

  // Constructor

  Service(            DEVICE_TYPE type,
              MISNET_CODE misnet_code,
              STATE state,
              ACCESS_TYPE access_type,
              REQUEST_MODE request_mode,
              UP_MODE up_mode,
              ACCESS_PIN access_pins[6],
              uint32_t subsample_rate,
              Value& delta_threshold,
              Value& low_threshold_limit,
              Value& high_threshold_limit,
              ACTION action,
              OUTPUT_MODE output_mode,
              std::string comment);


  virtual ~Service() { }

  virtual float readValue(void) = 0;
  
  DEVICE_TYPE getDeviceType() {
    return this->device_type;
  }
  void setDeviceType(DEVICE_TYPE type) {
     this->device_type = type;
   }
  MISNET_CODE getMisnetCode() {
    return this->misnet_code;
  }
  void setMisnetCode(MISNET_CODE code) {
     this->misnet_code = code;
   }
  STATE getState() {
    return this->state;
  }
  void setState(STATE state) {
    this->state = state;
  }
  ACCESS_TYPE getAccessType() {
    return this->access_type;
  }
  void setAccessType(ACCESS_TYPE type) {
    this->access_type = type;
  }
  REQUEST_MODE getRequestMode() {
    return this->request_mode;
  }
  void setRequestMode(REQUEST_MODE mode) {
      this->request_mode = mode;
  }
  UP_MODE getUpMode() {
    return this->up_mode;
  }
  void setUpMode(UP_MODE mode) {
    this->up_mode = mode;
  }
  ACCESS_PIN* getAccessPins() {
    return this->access_pins;
  }
  void setAccessPins(ACCESS_PIN* access_pins) {
          for (int i = 0; i < 6; i++) {
              this->access_pins[i] = access_pins[i];
          }
  }
  ACCESS_PIN getAccessPin(short index) {
    return this->access_pins[index - 1];
  }

  uint32_t getSubsampleRate() {
    return this->subsample_rate;
  }
  void setSubsampleRate(uint32_t rate) {
    this->subsample_rate = rate;
  }
  ACTION getAction() {
    return this->action;
  }
  void setAction(ACTION act) {
    this->action = act ;
  }
  OUTPUT_MODE getOutputMode() {
    return this->output_mode;
  }
  void setOutputMode(OUTPUT_MODE mode) {
    this->output_mode = mode;
  }
  std::string getComment() {
    return this->comment;
  }
  void setComment(std::string str) {
    this->comment =str;
  }
  uint32_t getActivationNb() {
    return this->activation_nb;
  }
  void setActivationNb(uint32_t act) {
    this->activation_nb = act;
  }


  // This method is used to check whether the service is ready to be sampled
  // (ENABLED state and subsampling rate OK)
  bool readyToSample() const {
    return ((this->activation_nb + 1) == this->subsample_rate);
  }

  // This method is used to process a heartbeat : it checks whether the service
  // is ready to be sampled, and updates accordingly the service activation number
  // (reset to 0 if the service is ready ti be sampled, otherwise increments it)
  bool processHeartbeat();

  void incrementActivationNb() {
    ++(this->activation_nb);
  }

  void resetActivationNb() {
    this->activation_nb = 0;
  }

  // This method writes the value passed as argument in the value field of the service.
  // The caller must write the value in "value" field of the srtucture AND set the
  // value of the "type" field of this structure.
  void setValue(Value& value) {
    this->previous_value = this->current_value;
    this->current_value = value;
  }

  // This other method returns the address of the area where sensor values must be written.
  // The caller must write the value in "value" field of the structure AND set the
  // value of the "type" field of this structure.
  // This method is quicker than the previous one since there is no neeed to copy the value
  // as in the previous method.
  // Drawback : the next method (savePreviousValue) must be called before this one,
  // so that the previous value may be saved. This is necessary when the UP_MODE is BY_RANGE
  // and the delta_threshold field is not empty.
  Value & getValueAddress() {
    return this->current_value;
  }

  void savePreviousValue() {
    this->previous_value = this->current_value;
  }

  // Check whether current value must be sent to gateway
  bool valueToBeSentToGateway();

  
  std::string toString() {
    std::ostringstream stringStream;
    stringStream << "Device type : " << this->device_type << ", MISNet code : " << (int) this->misnet_code;
    return stringStream.str();
  }

  // Returns a string representing a value. Used to build messages and to debug.
  // std::string toString(Value& value);

  // Returns a string representing the current value.
  std::string getCurrentValueAsString() {
    return this->current_value.toString();
  }

  // Returns a string representing the previous value.
  std::string getPreviousValueAsString() {
    return this->previous_value.toString();
  }

  // Set the delta threshold
  void setDeltaThreshold(Value delta) {
    this->delta_threshold = delta;
  }

  // Get the delta threshold
  Value getDeltaThreshold() {
    return this->delta_threshold;
  }

  // Set the low threshold limit
  void setLowThresholdLimit(Value low_limit) {
    this->low_threshold_limit = low_limit;
  }

  // Get the delta threshold
  Value getLowThresholdLimit() {
    return this->low_threshold_limit;
  }

  // Set the high threshold limit
  void setHighThresholdLimit(Value high_limit) {
    this->high_threshold_limit = high_limit;
  }

  // Get the delta threshold
  Value getHighThresholdLimit() {
    return this->high_threshold_limit;
  }


private:
  DEVICE_TYPE         device_type;
  MISNET_CODE         misnet_code;
  STATE               state;
  ACCESS_TYPE         access_type;
  ACCESS_PIN          access_pins[6];
  REQUEST_MODE        request_mode;
  UP_MODE             up_mode;
  uint32_t            subsample_rate;
  Value           delta_threshold;
  Value           low_threshold_limit;
  Value           high_threshold_limit;
  ACTION              action;
  OUTPUT_MODE         output_mode;
  std::string         comment;

  uint32_t      activation_nb;

  Value         current_value;
  Value         previous_value;
};

#endif // __SERVICE_HPP__
