/*
 * pcf8593_component.hpp
 *
 *  Created on: 18 sept. 2018
 *      Author: hoel
 */

#ifndef COMPONENTS_PCF8593_COMPONENT_HPP_
#define COMPONENTS_PCF8593_COMPONENT_HPP_

#include <iostream>
#include <cstdlib>
#include "mbed.h"
#include "Context.h"
#include "Service.hpp"
#include "Component.hpp"

using namespace std;
using namespace misnet;

#define LOCATION_CONTROL 			0x00
#define LOCATION_MILLISECONDS 		0x01
#define LOCATION_SECONDS 			0x02
#define LOCATION_MINUTES 			0x03
#define LOCATION_HOURS 				0x04
#define LOCATION_DAY 				0x05
#define LOCATION_MONTH 				0x06
#define LOCATION_YEAR 				0x07
#define LOCATION_ALARM_CONTROL 		0x08
#define LOCATION_ALARM_SECONDS 		0x10
#define LOCATION_COUNTER 			0x01
#define LOCATION_OFFSET_YEAR 		0x10
#define LOCATION_LAST_YEAR 			0x11

#define MODE_CLOCK_32KHZ 			0x00
#define MODE_CLOCK_50HZ 			0x10
#define MODE_EVENT_COUNTER 			0x20
#define MODE_TEST 					0x30

#define MONDAY 		0
#define TUESDAY 	1
#define WEDNESDAY 	2
#define THURSDAY 	3
#define FRIDAY 		4
#define SATURDAY 	5
#define SUNDAY 		6

#define BASE_YEAR 	2012
#define ALARM_ANY 	99

#define constrain(amt,low,high) ((amt)<(low)?(low):((amt)>(high)?(high):(amt)))

class pcf8593_component : public Component{

	private:
	uint8_t 	_address;
	I2C         *i2c_p;
	I2C         &i2c;
	char 		cmd[2];

	public:
	pcf8593_component(COMPONENT_ID id, vector<Service*>& services,PinName sda, PinName scl,uint8_t address) :
		i2c_p(new I2C(sda, scl)),
    	i2c(*i2c_p)
	{
		i2c.frequency(100000);
	    _address = address >> 1;  // convert to 7 bit so i2c doesn't choke
	}
	void reset() {

		//i2c.start();
		i2c.write(_address,(char*)LOCATION_CONTROL,1);
	    i2c.write(_address,(char*)0x04,1);    // 00 control/status (alarm enabled by default)
	    i2c.write(_address,(char*)0x00,1);    // 01 set hundreds-of-seconds
	    i2c.write(_address,(char*)0x00,1);    // 02 set second
	    i2c.write(_address,(char*)0x00,1);    // 03 set minute
	    i2c.write(_address,(char*)0x00,1);    // 04 set hour (24h format)
	    i2c.write(_address,(char*)0x01,1);    // 05 set day
	    i2c.write(_address,(char*)0x01,1);    // 06 set month
	    i2c.write(_address,(char*)0x00,1);    // 07 set timer
	    i2c.write(_address,(char*)0x00,1);    // 08 set alarm control
	    i2c.write(_address,(char*)0x00,1);    // 09 set alarm hundreds-of-seconds
	    i2c.write(_address,(char*)0x00,1);    // 0A set alarm second
	    i2c.write(_address,(char*)0x00,1);    // 0B set alarm minute
	    i2c.write(_address,(char*)0x00,1);    // 0C set alarm hour
	    i2c.write(_address,(char*)0x01,1);    // 0D set alarm day
	    i2c.write(_address,(char*)0x01,1);   // 0E set alarm month
	    i2c.write(_address,(char*)0x00,1);    // 0F set alarm timer
	    i2c.write(_address,(char*)0x00,1);    // 10 set year offset to 0
	    i2c.write(_address,(char*)0x00,1);    // 11 set last read value for year to 0
        //i2c.stop();
	}
	void reset(uint8_t mode) {
	    // TODO
	}
	void setMode(uint8_t mode) {
	    uint8_t control = getRegister(LOCATION_CONTROL);
	    control = (control & ~MODE_TEST) | (mode & MODE_TEST);
	    setRegister(LOCATION_CONTROL, control);
	}
	uint8_t getMode() {
	    return getRegister(LOCATION_CONTROL) & MODE_TEST;
	}
	int getMillisecond() {
	    return (int) bcd2byte(getRegister(LOCATION_MILLISECONDS)) * 10;
	}
	uint8_t getSecond() {
	    return bcd2byte(getRegister(LOCATION_SECONDS) & 0x7F);
	}
	uint8_t getMinute() {
	    return bcd2byte(getRegister(LOCATION_MINUTES) & 0x7F);
	}
	uint8_t getHour() {
	    return bcd2byte(getRegister(LOCATION_HOURS) & 0x3F);
	}
	unsigned long getTime() {
	    unsigned long total = 0;
	    total = getHour();
	    total = total * 60 + getMinute();
	    total = total * 60 + getSecond();
	    total = total * 1000 + getMillisecond();
	    return total;
	}
	char * getISOTime() {
	    // TODO
		return 0;
	}
	void setTime(uint8_t sec, uint8_t min, uint8_t hour) {
	    stop();
	    //i2c.start();
	    i2c.write(_address,(char*)LOCATION_SECONDS,1,1);
	    i2c.write(_address,(char*)byte2bcd(constrain(sec, 0, 59)),1);
	    i2c.write(_address,(char*)byte2bcd(constrain(min, 0, 59)),1);
	    i2c.write(_address,(char*)byte2bcd(constrain(hour, 0, 23)),1);
	    //i2c.stop();
	    start();
	}
	uint8_t getDay() {
	    return bcd2byte(getRegister(LOCATION_DAY) & 0x3F);
	}
	uint8_t getMonth() {
	    return bcd2byte(getRegister(LOCATION_MONTH) & 0x1F);
	}
	int getYear() {

		// To get the current year we have to look to 4 values :
		// 1) the top 2 bits of the 5th location
		// 2) the last value requested for those digits
		// 3) the offset value stored in the LOCATION_OFFSET_YEAR location
		// 4) the base year (2000)
	    uint8_t year = getRegister(LOCATION_DAY) >> 6;
	    uint8_t last = getRegister(LOCATION_LAST_YEAR);
	    uint8_t offset = getRegister(LOCATION_OFFSET_YEAR);

	    // we have changed the year: happy new year!!!
	    if (last != year) {

	        // we have overflowed the year value: update the offset
	        if (last > year) {
	            offset = (uint8_t) (offset + 4);
	            setRegister(LOCATION_OFFSET_YEAR, offset);
	        }
	        setRegister(LOCATION_LAST_YEAR, year);
	    }
	    return BASE_YEAR + offset + year;
	}
	uint8_t getWeekday() {
	    return getRegister(LOCATION_MONTH) >> 5;
	}
	char * getISODate() {
	    // TODO
		return 0;
	}
	void setDate(uint8_t day, uint8_t month, int year) {
	    setDate(day, month, year, 0);
	}
	void setDate(uint8_t day, uint8_t month, int year, uint8_t weekday) {

	    stop();

	    // Calculate the 2 bit value for the year on top of the
	    // BASE_YEAR and offset.
	    // (BASE_YEAR + offset) has to be a leap year
	    year -= BASE_YEAR;
	    uint8_t offset = year & 0xFC;
	    year -= offset;

	    uint8_t value;
	    //i2c.start();
	    i2c.write(_address,(char*)LOCATION_DAY,1,1);
	    value = byte2bcd(constrain(day, 1, 31)) | (year << 6);
	    i2c.write(_address,(char*)value,1,1);
	    value = byte2bcd(constrain(month, 1, 12)) | ((constrain(weekday, 0, 6)) << 5);
	    i2c.write(_address,(char*)value,1);
	    //i2c.stop();

	    //i2c.start();
	    i2c.write(_address,(char*)LOCATION_OFFSET_YEAR,1,1);
	    i2c.write(_address,(char*)offset,1,1);
	    i2c.write(_address,(char*)year,1);
	    //i2c.stop();

	    start();
	}
	void setDateTime(uint8_t sec, uint8_t min, uint8_t hour, uint8_t day, uint8_t month, int year) {
	    setDateTime(sec, min, hour, day, month, year, 0);
	}
	void setDateTime(uint8_t sec, uint8_t min, uint8_t hour, uint8_t day, uint8_t month, int year, uint8_t weekday) {
	    setTime(sec, min, hour);
	    setDate(day, month, year, weekday);
	}
	void setAlarm(uint8_t sec, uint8_t min, uint8_t hour) {
		// TODO : Only daily alarms support for now
	    //i2c.start();
	    i2c.write(_address,(char*)LOCATION_ALARM_SECONDS,1);
	    i2c.write(_address,(char*)byte2bcd(constrain(sec, 0, 59)),1,1);
	    i2c.write(_address,(char*)byte2bcd(constrain(min, 0, 59)),1,1);
	    i2c.write(_address,(char*)byte2bcd(constrain(hour, 0, 23)),1);
	    //i2c.stop();
	}
	void setCountAlarm(unsigned long count) {

	    // TODO
		//The event counter stores up to 6 digits of data, which are stored as 6 hexadecimal
		//values located in the registers 1h, 2h, and 3h. Therefore, up to 1 million events
		//may be recorded. An event counter alarm occurs when the event counter registers match
		//the value programmed in the registers 9h, Ah, and Bh, and the event alarm is enabled
		//(bits 4 and 5 which are logic 01 in the alarm control register). In this event,
		//the alarm flag (bit 1 of the control and status register) is set.
		//The inverted value of this flag can be transferred to the interrupt pin (pin 7)
		//by setting the alarm interrupt enable in the alarm control register. In
		//All information provided in this document is subject to legal disclaimers.
		//in this mode, the timer (location 07h) increments once for every one, one hundred,
		//ten thousand, or 1 million events, depending on the value programmed in bits 0, 1 and 2
		//of the alarm control register. In all other events, the timer functions are as in the clock mode.
	}
	void enableAlarm() {
	    uint8_t control;

	    control = getRegister(LOCATION_CONTROL);
	    control |= 0x04;
	    setRegister(LOCATION_CONTROL, control);

	    control = getRegister(LOCATION_ALARM_CONTROL);
	    control |= 0x90;
	    setRegister(LOCATION_ALARM_CONTROL, control);
	}
	void disableAlarm() {
	    uint8_t control = getRegister(LOCATION_ALARM_CONTROL);
	    control &= 0x6F;
	    setRegister(LOCATION_ALARM_CONTROL, control);
	}
	void clearInterrupt() {
	    uint8_t control = getRegister(LOCATION_ALARM_CONTROL);
	    control &= 0x6F;
	    setRegister(LOCATION_ALARM_CONTROL, control);
	}
	void setCount(unsigned long count) {
	    stop();
		i2c.write(_address, (char*)LOCATION_COUNTER, 1);
		i2c.write(_address, (char*)byte2bcd(count % 100), 1);
		i2c.write(_address, (char*)byte2bcd((count / 100) % 100), 1);
		i2c.write(_address, (char*)byte2bcd((count / 10000) % 100), 1);
		start();
	}
	unsigned long getCount() {

		i2c.write(_address, (char*)LOCATION_COUNTER, 1);
		i2c.read(_address, &cmd[0], 3);
	    unsigned long count = 0;
	    count = bcd2byte(cmd[0]);
	    count = count + bcd2byte(cmd[1]) * 100L;
	    count = count + bcd2byte(cmd[2]) * 10000L;
	    return count;
	}
	void stop() {
	    uint8_t control = getRegister(LOCATION_CONTROL);
	    control |= 0x80;
	    setRegister(LOCATION_CONTROL, control);
	}
	void start() {
	    uint8_t control = getRegister(LOCATION_CONTROL);
	    control &= 0x7F;
	    setRegister(LOCATION_CONTROL, control);
	}
	void setRegister(uint8_t offset, uint8_t value) {
	    i2c.start();
	    i2c.write(_address,(char*)offset,1);
	    i2c.write(_address,(char*)value,1);
	    i2c.stop();
	}
	uint8_t getRegister(uint8_t offset) {
	    i2c.write(_address, (char*)offset, 1);
	    i2c.read(_address, &cmd[0], 1);
	    return cmd[0];
	}
	uint8_t bcd2byte(uint8_t value){
	    return ((value >> 4) * 10) + (value & 0x0f);
	}
	uint8_t byte2bcd(uint8_t value){
	    return ((value / 10) << 4) + (value % 10);
	}
	void init(){
		// TODO
	}
	void readValues(void){
		for (std::vector<misnet::Service*>::iterator
				srvIt = this->getServices().begin();
				srvIt != this->getServices().end();
				srvIt++) {
			(*srvIt)->readValue();
		}
	}

	class pcf8593_value : public Service{

		public:
    	pcf8593_value(
				DEVICE_TYPE type,
				MISNET_CODE misnet_code,
				STATE state,
				ACCESS_TYPE access_type,
				REQUEST_MODE request_mode,
				UP_MODE up_mode,
				ACCESS_PIN access_pins[6],
				uint32_t subsample_rate,
				ACTION action,
				OUTPUT_MODE output_mode,
				string comment,
				pcf8593_component* parent
				) :
				Service(type, misnet_code, state, access_type, request_mode,
					    up_mode, access_pins, subsample_rate, action, output_mode, comment)
		{
			this->parent = parent;
		}
    	virtual ~pcf8593_value() {}
    	bool readValue(Service* service){
    		return false;
    	}

		private:
    	pcf8593_component * parent;
    };
};

#endif /* COMPONENTS_PCF8593_COMPONENT_HPP_ */
