/*
 * lsm303d_component.hpp
 *
 *  Created on: 18 sept. 2018
 *      Author: hoel
 */

#ifndef COMPONENTS_LSM303D_COMPONENT_HPP_
#define COMPONENTS_LSM303D_COMPONENT_HPP_

#include <iostream>
#include <cstdlib>
#include "mbed.h"
#include "Context.h"
#include "Service.hpp"
#include "Component.hpp"

using namespace std;
using namespace misnet;

#define MAG_ADDRESS  0x3C
#define ACC_ADDRESS  0x32

#define CTRL_REG1_A       0x20
#define CTRL_REG2_A       0x21
#define CTRL_REG3_A       0x22
#define CTRL_REG4_A       0x23
#define CTRL_REG5_A       0x24
#define CTRL_REG6_A       0x25 // DLHC only
#define REFERENCE_A       0x26
#define STATUS_REG_A      0x27

#define OUT_X_L_A         0x28
#define OUT_X_H_A         0x29
#define OUT_Y_L_A         0x2A
#define OUT_Y_H_A         0x2B
#define OUT_Z_L_A         0x2C
#define OUT_Z_H_A         0x2D

#define INT1_CFG_A        0x30
#define INT1_SRC_A        0x31
#define INT1_THS_A        0x32
#define INT1_DURATION_A   0x33
#define INT2_CFG_A        0x34
#define INT2_SRC_A        0x35
#define INT2_THS_A        0x36
#define INT2_DURATION_A   0x37

#define CRA_REG_M         0x00
#define CRB_REG_M         0x01
#define MR_REG_M          0x02

#define OUT_X_H_M         0x03
#define OUT_X_L_M         0x04
#define OUT_Y_H_M         0x07
#define OUT_Y_L_M         0x08
#define OUT_Z_H_M         0x05
#define OUT_Z_L_M         0x06

#define SR_REG_M          0x09
#define IRA_REG_M         0x0A
#define IRB_REG_M         0x0B
#define IRC_REG_M         0x0C

class LSM303DLHC_component : public Component{

	public:
	LSM303DLHC_component(COMPONENT_ID id, vector<Service*>& services, PinName sda, PinName scl):
		_device(sda, scl)
	{
		this->setServices(services);
		this->setId(id);
		_device.frequency(400000);
	}
	~LSM303DLHC_component();
	void init(){
		// init mag
		// continuous conversion mode
		_data[0] = MR_REG_M;
		_data[1] = 0x00;
		_device.write(MAG_ADDRESS, _data, 2);
		// data rate 75hz
		_data[0] = CRA_REG_M;
		_data[1] = 0x18; // 0b00011000
		_device.write(MAG_ADDRESS, _data, 2);
		// init acc
		// data rate 100hz
		_data[0] = CTRL_REG1_A;
		_data[1] = 0x2F; // 0b00101111
		_device.write(ACC_ADDRESS, _data, 2);
	}
	void readValues(void){
		for (std::vector<misnet::Service*>::iterator
				srvIt = this->getServices().begin();
				srvIt != this->getServices().end();
				srvIt++) {
			(*srvIt)->readValue();
		}
	}

	private:
	void setScale(float x, float y, float z){
		scale[0] = x;
		scale[1] = y;
		scale[2] = z;
	}
	void setOffset(float x, float y, float z){
		offset[0] = x;
		offset[1] = y;
		offset[2] = z;
	}

    I2C _device;
    char _data[6];
    int offset[3], scale[3];

    class LSM303DLHC_accelerometer : public Service{

		public:
		LSM303DLHC_accelerometer(
				DEVICE_TYPE type,
				MISNET_CODE misnet_code,
				STATE state,
				ACCESS_TYPE access_type,
				REQUEST_MODE request_mode,
				UP_MODE up_mode,
				ACCESS_PIN access_pins[6],
				uint32_t subsample_rate,
				ACTION action,
				OUTPUT_MODE output_mode,
				string comment,
				LSM303DLHC_component* parent
				) :
				Service(type, misnet_code, state, access_type, request_mode,
						up_mode, access_pins, subsample_rate, action, output_mode, comment)
		{
			this->parent = parent;
		}
		virtual ~LSM303DLHC_accelerometer() {}
		bool readValue(void) {
			int a[3];
			parent->_data[0] = OUT_X_L_A | (1<<7);
			parent->_device.write(ACC_ADDRESS, _data, 1);
			parent->_device.read(ACC_ADDRESS, _data, 6);
			// 12-bit values
			a[0] = (short)(parent->_data[1]<<8 | parent->_data[0]) >> 4;
			a[1] = (short)(parent->_data[3]<<8 | parent->_data[2]) >> 4;
			a[2] = (short)(parent->_data[5]<<8 | parent->_data[4]) >> 4;
            //this->getValueAddress().set) // triple int value
            this->savePreviousValue();
			return false;
		}

		private:
		LSM303DLHC_component *parent;
    };

    class LSM303DLHC_megnetometer : public Service{

		public:
		LSM303DLHC_megnetometer(
				DEVICE_TYPE type,
				MISNET_CODE misnet_code,
				STATE state,
				ACCESS_TYPE access_type,
				REQUEST_MODE request_mode,
				UP_MODE up_mode,
				ACCESS_PIN access_pins[6],
				uint32_t subsample_rate,
				ACTION action,
				OUTPUT_MODE output_mode,
				string comment,
				LSM303DLHC_component* parent
				) :
				Service(type, misnet_code, state, access_type, request_mode,
					    up_mode, access_pins, subsample_rate, action, output_mode, comment)
		{
			this->parent = parent;
		}
		virtual ~LSM303DLHC_megnetometer() {}
		bool readValue(void) {
			int a[3];
			parent->_data[0] = OUT_X_H_M;
			parent->_device.write(MAG_ADDRESS, parent->_data, 1);
			parent->_device.read(MAG_ADDRESS, parent->_data, 6);
			a[0] = (short) (parent->_data[0]<<8 | parent->_data[1]); // X
			a[1] = (short) (parent->_data[4]<<8 | parent->_data[5]); // Y
			a[2] = (short) (parent->_data[2]<<8 | parent->_data[3]); // Z
            //this->getValueAddress().set) // triple int value
            this->savePreviousValue();
			return false;
		}

		private:
		LSM303DLHC_component *parent;
    };
};

#endif /* COMPONENTS_LSM303D_COMPONENT_HPP_ */
