#ifndef __VALUE_HPP__
#define __VALUE_HPP__

#include "Context.h"

#ifndef TEST_ENVIRONMENT
#include "mbed.h"
#endif

#ifdef CPLUSPLUS_99
#include <stdint.h>
#endif

#include <iostream>
#include <sstream>
#include <string>


namespace misnet {
    class Value;
}


class misnet::Value {

    public:
        enum VALUE_TYPE {
            NOT_SET         = 0,    // Not initialised
            BOOLEAN         = 1,    // Who knows ?
            CHAR            = 2,    // Single character
            UINT8_T         = 3,
            INT8_T          = 4,
            UINT16_T        = 5,
            INT16_T         = 6,
            UINT32_T        = 7,
            INT32_T         = 8,
            FLOAT           = 9,    // Floating value on 32 bits
            DOUBLE          = 10,   // Floating value on 64 bits
            TIME            = 11,   // Unix time
            TRIPLE_DOUBLE   = 12    // Used for accelerometer values
                            // or other values of the same kind
        } ;

        typedef struct {
          union {
            bool        bool_value;
            char        char_value;
            uint8_t     uint8_value;
            int8_t      int8_value;
            uint16_t        uint16_value;
            int16_t     int16_value;
            uint32_t        uint32_value;
            int32_t     int32_value;
            float       float_value;
            double      double_value;
            uint32_t        time_value;
            double      triple_double_values[3];
          } value;
          VALUE_TYPE type;
        } GENERIC_VALUE;


  // Default constructor
  Value() {
    this->_value.type = NOT_SET;
  }

  Value(GENERIC_VALUE value) : _value(value) {
  }


  // Setters
  // -------
  void setValue(GENERIC_VALUE value);

  void setBoolValue(bool value) {
    this->_value.type = BOOLEAN;
    this->_value.value.bool_value = value;
  }

  void setCharValue(char value) {
    this->_value.type = CHAR;
    this->_value.value.char_value = value;
  }

  void setUint8Value(uint8_t value) {
    this->_value.type = UINT8_T;
    this->_value.value.uint8_value = value;
  }

  void setInt8Value(int8_t value) {
    this->_value.type = INT8_T;
    this->_value.value.int8_value = value;
  }

  void setUint16Value(uint16_t value) {
    this->_value.type = UINT16_T;
    this->_value.value.uint16_value = value;
  }

  void setInt16Value(int16_t value) {
    this->_value.type = INT16_T;
    this->_value.value.int16_value = value;
  }

  void setUint32Value(uint32_t value) {
    this->_value.type = UINT32_T;
    this->_value.value.uint32_value = value;
  }

  void setInt32Value(int32_t value) {
    this->_value.type = INT32_T;
    this->_value.value.int32_value = value;
  }

  void setFloatValue(float value) {
    this->_value.type = FLOAT;
    this->_value.value.float_value = value;
  }

  void setDoubleValue(double value) {
    this->_value.type = DOUBLE;
    this->_value.value.double_value = value;
  }

  void setTimeValue(uint32_t value) {
    this->_value.type = TIME;
    this->_value.value.time_value = value;
  }

  void setTripleDoubleValue(double value1, double value2, double value3) {
    this->_value.type = DOUBLE;
    this->_value.value.triple_double_values[0] = value1;
    this->_value.value.triple_double_values[1] = value2;
    this->_value.value.triple_double_values[2] = value3;
  }


  // Getters
  // -------
  VALUE_TYPE getType() {
    return this->_value.type;
  }

  bool getBoolValue() {
    return this->_value.value.bool_value;
  }

  char getCharValue() {
    return this->_value.value.char_value;
  }

  uint8_t getUint8Value() {
    return this->_value.value.uint8_value;
  }

  int8_t getInt8Value() {
    return this->_value.value.int8_value;
  }

  uint16_t getUint16Value() {
    return this->_value.value.uint16_value;
  }

  int16_t getInt16Value() {
    return this->_value.value.int16_value;
  }

  uint32_t getUint32Value() {
    return this->_value.value.uint32_value;
  }

  int32_t getInt32Value() {
    return this->_value.value.int32_value;
  }

  float getFloatValue() {
    return this->_value.value.float_value;
  }

  double getDoubleValue() {
    return this->_value.value.double_value;
  }

  uint32_t getTimeValue() {
    return this->_value.value.time_value;
  }

  double* getTripleDoubleValue() {
    return this->_value.value.triple_double_values;
  }


  /* This method returns :
  -2 if at least one of the two Values is not set, or if both values don't hold the same type
  -1 if the Value object is lower than the parameter Value
  0 if the Value object is equal to the parameter Value
  1 if the Value object is greater than the parameter Value

  NB. For boolean Values, true > false.
  NB. For float and double values, equality is somewhat hazardous due to precision.
  */
  int8_t compareTo(Value & otherValue);


  Value substract(Value& otherValue);


  /* This methode substracts a value from the current one and checks whether the result
     absolute value is greather than a delta or not.
     In case the absolute value is greater, returns true, otherwise return false.
     NB. In case the various values don't hold the same type, returns true. This way, when
     a terminal is started and there is no "previous value", this method will return true,
     so that the sensor can send its initial value.
  */
  bool isAbsoluteDifferenceValueGreaterThanDelta(Value& valueToSubstract, Value& delta);


  /* This method returns a boolean value telling whether or not the current value
     differs from the parameter value.
  */
  bool isDifferentFrom(Value& otherValue);


  std::string toString();


  GENERIC_VALUE     _value;
};

#endif // __VALUE_HPP__
