/*
 * DS18B20.hpp
 *
 *  Created on: 18 sept. 2018
 *      Author: hoel
 */

#ifndef DS18B20_COMPONENT_HPP_
#define DS18B20_COMPONENT_HPP_

#include <iostream>
#include <cstdlib>
#include "mbed.h"
#include "Context.h"
#include "Service.hpp"
#include "Component.hpp"
#include "LinkedList/LinkedList.h"

using namespace std;
using namespace misnet;

#define FAMILY_CODE _ROM[0]
#define FAMILY_CODE_DS1820 	0x10
#define FAMILY_CODE_DS18B20 0x28
#define FAMILY_CODE_DS1822  0x22
#define ONEWIRE_INPUT(pin)  pin->input()
#define ONEWIRE_OUTPUT(pin) pin->output()
#define ONEWIRE_INIT(pin)
#define INIT_DELAY
#define ONEWIRE_DELAY_US(value) wait_us(value)

class ds18b20_component : public Component{

	private:
	bool _parasite_power;
	bool _power_mosfet;
	bool _power_polarity;
	char _ROM[8],RAM[9];
	DigitalInOut _datapin;
	DigitalOut _parasitepin;
	static LinkedList<node> probes;
	bool crcerr = 0;

	public:
    enum devices{
        this_device,     // command applies to only this device
        all_devices };   // command applies to all devices
    enum { invalid_conversion = -1000 };
    ds18b20_component (COMPONENT_ID id, vector<Service*>& services, PinName data_pin, PinName power_pin=NC, bool power_polarity=NC) :
    	_datapin(data_pin),
		_parasitepin(power_pin)
    {
		this->setServices(services);
		this->setId(id);
    	if(power_pin) _parasite_power = true;
        _power_polarity = power_polarity;
        _power_mosfet = power_pin != NC;
    }
    ~ds18b20_component();
    bool init(void){

    	int byte_counter;
    	for(byte_counter=0;byte_counter<9;byte_counter++)
            RAM[byte_counter] = 0x00;

        ONEWIRE_INIT((&_datapin));
        INIT_DELAY;

        if (!unassignedProbe(&_datapin, _ROM))
            error("No unassigned DS18B20_component found!\n");
        else {
            _datapin.input();
            probes.append(this);
            _parasite_power = !read_power_supply(devices::this_device);
        }
        return false;
    }
    bool onewire_reset(DigitalInOut *pin) {
    // This will return false if no devices are present on the data bus
        bool presence=false;
        ONEWIRE_OUTPUT(pin);
        pin->write(0);          // bring low for 500 us
        ONEWIRE_DELAY_US(500);
        ONEWIRE_INPUT(pin);       // let the data line float high
        ONEWIRE_DELAY_US(90);            // wait 90us
        if (pin->read()==0) // see if any devices are pulling the data line low
            presence=true;
        ONEWIRE_DELAY_US(410);
        return presence;
    }
    void onewire_bit_out (DigitalInOut *pin, bool bit_data) {
        ONEWIRE_OUTPUT(pin);
        pin->write(0);
        ONEWIRE_DELAY_US(3);                 // DXP modified from 5
        if (bit_data) {
            pin->write(1); // bring data line high
            ONEWIRE_DELAY_US(55);
        } else {
            ONEWIRE_DELAY_US(55);            // keep data line low
            pin->write(1);
            ONEWIRE_DELAY_US(10);            // DXP added to allow bus to float high before next bit_out
        }

    }
    void onewire_byte_out(char data) { // output data character (least sig bit first).
        int n;
        //printf("Byte out 0x%02x\n", data);
        for (n=0; n<8; n++) {
            onewire_bit_out(&this->_datapin, data & 0x01);
            data = data >> 1; // now the next bit is in the least sig bit position.
        }

    }
    bool onewire_bit_in(DigitalInOut *pin) {
        bool answer;
        ONEWIRE_OUTPUT(pin);
        pin->write(0);
        ONEWIRE_DELAY_US(3);                 // DXP modofied from 5
        ONEWIRE_INPUT(pin);
        ONEWIRE_DELAY_US(10);                // DXP modified from 5
        answer = pin->read();
        ONEWIRE_DELAY_US(45);                // DXP modified from 50
        return answer;
    }
    char onewire_byte_in() { // read byte, least sig byte first
        char answer = 0x00;
        int i;
        for (i=0; i<8; i++) {
            answer = answer >> 1; // shift over to make room for the next bit
            if (onewire_bit_in(&this->_datapin))
                answer = answer | 0x80; // if the data port is high, make this bit a 1
        }
        //printf("Byte in 0x%02x\n", answer);
        return answer;
    }
    bool unassignedProbe(PinName pin) {
        DigitalInOut _pin(pin);
        ONEWIRE_INIT((&_pin));
        INIT_DELAY;
        char ROM_address[8];
        return search_ROM_routine(&_pin, 0xF0, ROM_address);
    }
    bool unassignedProbe(DigitalInOut *pin, char *ROM_address) {
        return search_ROM_routine(pin, 0xF0, ROM_address);
    }
    bool search_ROM_routine(DigitalInOut *pin, char command, char *ROM_address) {
        bool DS1820_done_flag = false;
        int DS1820_last_descrepancy = 0;
        char DS1820_search_ROM[8] = {0, 0, 0, 0, 0, 0, 0, 0};

        int descrepancy_marker, ROM_bit_index;
        bool return_value, Bit_A, Bit_B;
        unsigned char byte_counter;
        char bit_mask;

        return_value=false;
        while (!DS1820_done_flag) {
            if (!onewire_reset(pin)) {
                return false;
            } else {
                ROM_bit_index=1;
                descrepancy_marker=0;
                char command_shift = command;
                for (int n=0; n<8; n++) {           // Search ROM command or Search Alarm command
                    onewire_bit_out(pin, command_shift & 0x01);
                    command_shift = command_shift >> 1; // now the next bit is in the least sig bit position.
                }
                byte_counter = 0;
                bit_mask = 0x01;
                while (ROM_bit_index<=64) {
                    Bit_A = onewire_bit_in(pin);
                    Bit_B = onewire_bit_in(pin);
                    if (Bit_A & Bit_B) {
                        descrepancy_marker = 0; // data read error, this should never happen
                        ROM_bit_index = 0xFF;
                    } else {
                        if (Bit_A | Bit_B) {
                            // Set ROM bit to Bit_A
                            if (Bit_A) {
                                DS1820_search_ROM[byte_counter] = DS1820_search_ROM[byte_counter] | bit_mask; // Set ROM bit to one
                            } else {
                                DS1820_search_ROM[byte_counter] = DS1820_search_ROM[byte_counter] & ~bit_mask; // Set ROM bit to zero
                            }
                        } else {
                            // both bits A and B are low, so there are two or more devices present
                            if ( ROM_bit_index == DS1820_last_descrepancy ) {
                                DS1820_search_ROM[byte_counter] = DS1820_search_ROM[byte_counter] | bit_mask; // Set ROM bit to one
                            } else {
                                if ( ROM_bit_index > DS1820_last_descrepancy ) {
                                    DS1820_search_ROM[byte_counter] = DS1820_search_ROM[byte_counter] & ~bit_mask; // Set ROM bit to zero
                                    descrepancy_marker = ROM_bit_index;
                                } else {
                                    if (( DS1820_search_ROM[byte_counter] & bit_mask) == 0x00 )
                                        descrepancy_marker = ROM_bit_index;
                                }
                            }
                        }
                        onewire_bit_out (pin, DS1820_search_ROM[byte_counter] & bit_mask);
                        ROM_bit_index++;
                        if (bit_mask & 0x80) {
                            byte_counter++;
                            bit_mask = 0x01;
                        } else {
                            bit_mask = bit_mask << 1;
                        }
                    }
                }
                DS1820_last_descrepancy = descrepancy_marker;
                if (ROM_bit_index != 0xFF) {
                    int i = 1;
                    node *list_container;
                    while(1) {
                        list_container = probes.pop(i);
                        if (list_container == NULL) {                             //End of list, or empty list
                            if (ROM_checksum_error(DS1820_search_ROM)) {          // Check the CRC
                                return false;
                            }
                            for(byte_counter=0;byte_counter<8;byte_counter++)
                                ROM_address[byte_counter] = DS1820_search_ROM[byte_counter];
                            return true;
                        } else {                    //Otherwise, check if ROM is already known
                            bool equal = true;
                            ds18b20_component *pointer = (ds18b20_component*) list_container->data;
                            char *ROM_compare = pointer->_ROM;

                            for(byte_counter=0;byte_counter<8;byte_counter++) {
                                if ( ROM_compare[byte_counter] != DS1820_search_ROM[byte_counter])
                                    equal = false;
                            }
                            if (equal)
                                break;
                            else
                                i++;
                        }
                    }
                }
            }
            if (DS1820_last_descrepancy == 0)
                DS1820_done_flag = true;
        }
        return return_value;
    }
    void match_ROM() {
    	// select specific device
        int i;
        onewire_reset(&this->_datapin);
        onewire_byte_out( 0x55);  //Match ROM command
        for (i=0;i<8;i++) {
            onewire_byte_out(_ROM[i]);
        }
    }
    void skip_ROM() {
        onewire_reset(&this->_datapin);
        onewire_byte_out(0xCC);   // Skip ROM command
    }
    bool ROM_checksum_error(char *_ROM_address) {
        char _CRC=0x00;
        int i;
        for(i=0;i<7;i++) // Only going to shift the lower 7 bytes
            _CRC = CRC_byte(_CRC, _ROM_address[i]);
        // After 7 bytes CRC should equal the 8th byte (ROM CRC)
        return (_CRC!=_ROM_address[7]); // will return true if there is a CRC checksum mis-match
    }
    bool RAM_checksum_error() {
        char _CRC=0x00;
        int i;
        for(i=0;i<8;i++) // Only going to shift the lower 8 bytes
            _CRC = CRC_byte(_CRC, RAM[i]);
        // After 8 bytes CRC should equal the 9th byte (RAM CRC)
        return (_CRC!=RAM[8]); // will return true if there is a CRC checksum mis-match
    }
    char CRC_byte (char _CRC, char byte ) {
        int j;
        for(j=0;j<8;j++) {
            if ((byte & 0x01 ) ^ (_CRC & 0x01)) {
                // DATA ^ LSB CRC = 1
                _CRC = _CRC>>1;
                // Set the MSB to 1
                _CRC = _CRC | 0x80;
                // Check bit 3
                if (_CRC & 0x04) {
                    _CRC = _CRC & 0xFB; // Bit 3 is set, so clear it
                } else {
                    _CRC = _CRC | 0x04; // Bit 3 is clear, so set it
                }
                // Check bit 4
                if (_CRC & 0x08) {
                    _CRC = _CRC & 0xF7; // Bit 4 is set, so clear it
                } else {
                    _CRC = _CRC | 0x08; // Bit 4 is clear, so set it
                }
            } else {
                // DATA ^ LSB CRC = 0
                _CRC = _CRC>>1;
                // clear MSB
                _CRC = _CRC & 0x7F;
                // No need to check bits, with DATA ^ LSB CRC = 0, they will remain unchanged
            }
            byte = byte>>1;
        }
        return _CRC;
    }
    int convertTemperature(bool wait, devices device) {

        // Convert temperature into scratchpad RAM for all devices at once
        int delay_time = 750; // Default delay time
        char resolution;
        if (device==all_devices) skip_ROM(); // Skip ROM command, will convert for ALL devices
        else {
            match_ROM();
            if ((FAMILY_CODE == FAMILY_CODE_DS18B20 ) || (FAMILY_CODE == FAMILY_CODE_DS1822 )) {
                resolution = RAM[4] & 0x60;
                if (resolution == 0x00) // 9 bits
                    delay_time = 94;
                if (resolution == 0x20) // 10 bits
                    delay_time = 188;
                if (resolution == 0x40) // 11 bits
                    delay_time = 375;
                // 12bits uses the 750ms default
            }
        }
        onewire_byte_out( 0x44);  // perform temperature conversion
        if (_parasite_power) {
            if (_power_mosfet) {
                _parasitepin = _power_polarity;     // Parasite power strong pullup
                wait_ms(delay_time);
                _parasitepin = !_power_polarity;
                delay_time = 0;
            } else {
                _datapin.output();
                _datapin.write(1);
                wait_ms(delay_time);
                _datapin.input();
            }
        } else {
            if (wait) {
                wait_ms(delay_time);
                delay_time = 0;
            }
        }
        return delay_time;
    }
    void read_RAM() {
        // copy the DS18B20_component's 9 bytes of RAM data to RAM array
        int i;
        this->crcerr = 0;
        match_ROM();             // Select this device
        onewire_byte_out( 0xBE);   //Read Scratchpad command
        for(i=0;i<9;i++) {
            RAM[i] = onewire_byte_in();
        }
        if(!RAM_checksum_error()) this->crcerr = 1; // TODO check
    }
    bool setResolution(unsigned int resolution) {
        bool answer = false;
        resolution = resolution - 9;
        if (resolution < 4) {
            resolution = resolution<<5; // align the bits
            RAM[4] = (RAM[4] & 0x60) | resolution; // mask out old data, insert new
            write_scratchpad ((RAM[2]<<8) + RAM[3]);
            //store_scratchpad (DS18B20_component::this_device); // Need to test if this is required
            answer = true;
        }
        return answer;
    }
    void write_scratchpad(int data) {
        RAM[3] = data;
        RAM[2] = data>>8;
        match_ROM();
        onewire_byte_out(0x4E);   // Copy scratchpad into DS18B20_component ram memory
        onewire_byte_out(RAM[2]); // T(H)
        onewire_byte_out(RAM[3]); // T(L)
        if ((FAMILY_CODE == FAMILY_CODE_DS18B20 ) || (FAMILY_CODE == FAMILY_CODE_DS1822 )) {
            onewire_byte_out(RAM[4]); // Configuration register
        }
    }
    float temperature(char scale) {
    	// The data specs state that count_per_degree should be 0x10 (16), I found my devices
    	// to have a count_per_degree of 0x4B (75). With the standard resolution of 1/2 deg C
    	// this allowed an expanded resolution of 1/150th of a deg C. I wouldn't rely on this
    	// being super acurate, but it does allow for a smooth display in the 1/10ths of a
    	// deg C or F scales.
        float answer, remaining_count, count_per_degree;
        int reading;
        read_RAM();
        if (RAM_checksum_error()){
            // Indicate we got a CRC error
            answer = invalid_conversion;
            // TODO return true on crc error
        }
        else
        {
            reading = (RAM[1] << 8) + RAM[0];
            if (reading & 0x8000) { // negative degrees C
                reading = 0-((reading ^ 0xffff) + 1); // 2's comp then convert to signed int
            }
            answer = reading +0.0; // convert to floating point
            if ((FAMILY_CODE == FAMILY_CODE_DS18B20 ) || (FAMILY_CODE == FAMILY_CODE_DS1822 )) {
                answer = answer / 16.0f;
            }
            else {
                remaining_count = RAM[6];
                count_per_degree = RAM[7];
                answer = floor(answer/2.0f) - 0.25f + (count_per_degree - remaining_count) / count_per_degree;
            }
            if (scale=='F' or scale=='f')
                // Convert to deg F
                answer = answer * 9.0f / 5.0f + 32.0f;
        }
        return answer;
    }
    bool read_power_supply(devices device) {
    	// This will return true if the device (or all devices) are Vcc powered
    	// This will return false if the device (or ANY device) is parasite powered
        if (device==all_devices)
            skip_ROM();          // Skip ROM command, will poll for any device using parasite power
        else
            match_ROM();
        onewire_byte_out(0xB4);   // Read power supply command
        return onewire_bit_in(&this->_datapin);
    }
	void readValues(void){
		for (std::vector<misnet::Service*>::iterator
				srvIt = this->getServices().begin();
				srvIt != this->getServices().end();
				srvIt++) {
			(*srvIt)->readValue();
		}
	}

	class ds18b20_temperature : public Service{

    	public:
    	ds18b20_temperature(
    			DEVICE_TYPE type,
				MISNET_CODE misnet_code,
				STATE state,
				ACCESS_TYPE access_type,
				REQUEST_MODE request_mode,
				UP_MODE up_mode,
				ACCESS_PIN access_pins[6],
				uint32_t subsample_rate,
				ACTION action,
				OUTPUT_MODE output_mode,
				string comment,
				ds18b20_component* parent
				) :
			    Service(type, misnet_code, state, access_type, request_mode,
			            up_mode, access_pins, subsample_rate, action, output_mode, comment)
    	{
    		this->parent = parent;
    	}
    	virtual ~ds18b20_temperature() {}
    	void readerCb(){

    		float val = parent->temperature('c');
    		if(parent->crcerr) { return; } // TODO implement error handling
            this->getValueAddress().setFloatValue(val);
            this->savePreviousValue();
    	}
    	bool readValue(void) {

    		int delay = parent->convertTemperature(false, this_device);
    		reader.attach(callback(this, &ds18b20_component::ds18b20_temperature::readerCb),delay + 50);
    		return false; // no error handling on request
    	}

		private:
		ds18b20_component *parent;
		Timeout reader;
	};
};

#endif /* DS18B20_COMPONENT_HPP_ */
