/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "WEIGHTService.h"
#include "pretty_printer.h"
#include "Hx711.h"

Hx711 scale(D4, D5);  //D4 is connected to clock. D5 is conected to data.
float calibration_factor = -1.0f;

Serial pc(USBTX, USBRX);    // USB Serial Terminal

const static char DEVICE_NAME[] = "TH2G_WEIGHT";

static EventQueue event_queue(/* event count */ 16 * EVENTS_EVENT_SIZE);

class WEIGHTDemo : ble::Gap::EventHandler {
public:
    WEIGHTDemo(BLE &ble, events::EventQueue &event_queue) :
        _ble(ble),
        _event_queue(event_queue),
        _alive_led(LED1, 0),
        _weight_uuid(WEIGHTService::WEIGHT_SERVICE_UUID),
        _weight_service(NULL),
        _adv_data_builder(_adv_buffer) { }

//    ~WEIGHTDemo() {
//        delete _weight_service;
//    }

    void start() {
        _ble.gap().setEventHandler(this);

        _ble.init(this, &WEIGHTDemo::on_init_complete);

        _event_queue.call_every(1000, this, &WEIGHTDemo::update_sensor_value);

        _event_queue.dispatch_forever();
    }

private:
    /** Callback triggered when the ble initialization process has finished */
    void on_init_complete(BLE::InitializationCompleteCallbackContext *params) {
        if (params->error != BLE_ERROR_NONE) {
            printf("Ble initialization failed.");
            return;
        }
        
        initialWeightReading(&_initial_weight);

        _weight_service = new WEIGHTService(_ble, _initial_weight);

//        _ble.gattServer().onDataWritten(this, &WEIGHTDemo::on_data_written);

        print_mac_address();
        
        pc.printf("HX711 Demo\n");
        pc.printf("Ensure that there is no weight on the floor.\n");

        
        int offset = (int)scale.read();
        scale.set_offset(offset);
        printf("Reading: %d \n",offset);
        scale.set_scale(calibration_factor);

        start_advertising();
    }

    void start_advertising() {
        /* Create advertising parameters and payload */

        ble::AdvertisingParameters adv_parameters(
            ble::advertising_type_t::CONNECTABLE_UNDIRECTED,
            ble::adv_interval_t(ble::millisecond_t(1000))
        );

        _adv_data_builder.setFlags();
        _adv_data_builder.setLocalServiceList(mbed::make_Span(&_weight_uuid, 1));
        _adv_data_builder.setName(DEVICE_NAME);

        /* Setup advertising */

        ble_error_t error = _ble.gap().setAdvertisingParameters(
            ble::LEGACY_ADVERTISING_HANDLE,
            adv_parameters
        );

        if (error) {
            printf("_ble.gap().setAdvertisingParameters() failed\r\n");
            return;
        }

        error = _ble.gap().setAdvertisingPayload(
            ble::LEGACY_ADVERTISING_HANDLE,
            _adv_data_builder.getAdvertisingData()
        );

        if (error) {
            printf("_ble.gap().setAdvertisingPayload() failed\r\n");
            return;
        }

        /* Start advertising */

        error = _ble.gap().startAdvertising(ble::LEGACY_ADVERTISING_HANDLE);

        if (error) {
            printf("_ble.gap().startAdvertising() failed\r\n");
            return;
        }
    }

    void update_sensor_value() 
    {
        if (_ble.gap().getState().connected) 
        {
                _alive_led = !_alive_led;
                _current_weight = scale.read();
//                _current_weight = 456.87;
                WeightType_t weight;
                weight.integer = (int32_t)_current_weight;
                float f_decimal = _current_weight - weight.integer;
                weight.decimal = (uint8_t)(f_decimal* 100.0f);
                if(_current_weight <= 0.00f)
                {
                    _current_weight = 0.00;
                }
                _weight_service->updateWeight(_current_weight);
                pc.printf("Reading of float: %.2f\n", _current_weight);
                pc.printf("Reading of structure: %d.%d\n", weight.integer,weight.decimal);

         }
    }

private:
    /* Event handler */

    void onDisconnectionComplete(const ble::DisconnectionCompleteEvent&) {
        _ble.gap().startAdvertising(ble::LEGACY_ADVERTISING_HANDLE);
    }
    void initialWeightReading(WeightType_t *newValue){
        newValue->integer = 0;
        newValue->decimal = 0;    
    }

private:
    BLE &_ble;
    events::EventQueue &_event_queue;
    DigitalOut _alive_led;
    UUID _weight_uuid;
    
    float _current_weight;
    WeightType_t _initial_weight;
    WEIGHTService *_weight_service;

    uint8_t _adv_buffer[ble::LEGACY_ADVERTISING_MAX_SIZE];
    ble::AdvertisingDataBuilder _adv_data_builder;
};

/** Schedule processing of events from the BLE middleware in the event queue. */
void schedule_ble_events(BLE::OnEventsToProcessCallbackContext *context) {
    event_queue.call(Callback<void()>(&context->ble, &BLE::processEvents));
}

int main()
{
    //scale.setScale(calibration_factor); //This value is obtained by using the SparkFun_HX711_Calibration sketch
    //scale.tare(); 
    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(schedule_ble_events);

    WEIGHTDemo demo(ble, event_queue);
    demo.start();

    return 0;
}

