/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <events/mbed_events.h>

#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "ADS1256Service.h"
#include "pretty_printer.h"

#include "ADS1256.h"

#define SPI_MOSI  D11
#define SPI_MISO  D12
#define SPI_SCLK  D13
#define CHIP_SLCT D9
#define CHANNEL_NUM 4

DigitalIn ndrdy =  D10;
DigitalOut cs = CHIP_SLCT;


SPI spi(SPI_MOSI, SPI_MISO, SPI_SCLK);
ADS1256 ads(&spi, &ndrdy, &cs);

int32_t ads_sum = 0;
float calibration_factor = 12.09;
float weight_value=0;

const static char DEVICE_NAME[] = "ADS1256";

static EventQueue event_queue(/* event count */ 32 * EVENTS_EVENT_SIZE);


class ADS1256Demo : ble::Gap::EventHandler {
public:
    ADS1256Demo(BLE &ble, events::EventQueue &event_queue) :
        _ble(ble),
        _event_queue(event_queue),
        _led1(LED1, 0),
        _ads1256_service(NULL),
        _ads1256_uuid(ADS1256Service::ADS1256_SERVICE_UUID),
        _adv_data_builder(_adv_buffer) { }

    void start() {
        _ble.gap().setEventHandler(this);

        _ble.init(this, &ADS1256Demo::on_init_complete);

        _event_queue.call_every(1000, this, &ADS1256Demo::ads1256Readings);

        _event_queue.dispatch_forever();
    }

private:
    /** Callback triggered when the ble initialization process has finished */
    void on_init_complete(BLE::InitializationCompleteCallbackContext *params) {
        if (params->error != BLE_ERROR_NONE) {
            printf("Ble initialization failed.");
            return;
        }

        print_mac_address();
        
        /* @brief Set the configuration parameters:
            channel == 0(0x01h),
            PGA gain == 64,
            Buffer == 1 (enabled),
            Datarate == 2.5SPS.
            Auto_calibration has also been enabled
         */
        ads.cfgADC();
        printf("Device configuration is successful\n");
        ads.setDiffChannel(); // Redundancy feature just toensure that the channel is set correctly.
        printf("The set gain value is %d\n", ads.getGainVal());
        
        /*TODO: Calibration*/
        ads.selfCal();
        ads.sysOffCal();
        
        setInitialValue();

        /* Setup primary service. */
        _ads1256_service = new ADS1256Service(_ble, initialValue /* initial value of ads1256 */);

        start_advertising();
    }

    void start_advertising() {
        /* Create advertising parameters and payload */

        ble::AdvertisingParameters adv_parameters(
            ble::advertising_type_t::CONNECTABLE_UNDIRECTED,
            ble::adv_interval_t(ble::millisecond_t(1000))
        );

        _adv_data_builder.setFlags();
        _adv_data_builder.setLocalServiceList(mbed::make_Span(&_ads1256_uuid, 1));
        _adv_data_builder.setName(DEVICE_NAME);

        /* Setup advertising */

        ble_error_t error = _ble.gap().setAdvertisingParameters(
            ble::LEGACY_ADVERTISING_HANDLE,
            adv_parameters
        );

        if (error) {
            print_error(error, "_ble.gap().setAdvertisingParameters() failed");
            return;
        }

        error = _ble.gap().setAdvertisingPayload(
            ble::LEGACY_ADVERTISING_HANDLE,
            _adv_data_builder.getAdvertisingData()
        );

        if (error) {
            print_error(error, "_ble.gap().setAdvertisingPayload() failed");
            return;
        }

        /* Start advertising */

        error = _ble.gap().startAdvertising(ble::LEGACY_ADVERTISING_HANDLE);

        if (error) {
            print_error(error, "_ble.gap().startAdvertising() failed");
            return;
        }
    }

    void ads1256Readings(void) {
        _led1 = 1;
        float dummyValue= 12.4524;
        readDataMux();
        printf("The weight is: %0.2f grams\n", weight_value);
        _ads1256_service->updateAds1256State(weight_value);
    }
    void setInitialValue(){
        initialValue.integer = 0;
        initialValue.decimal = 0;
    }
    void readDataMux()
    {
        ads.readDiffChannel(ads.channel);    
        weight_value = ads.adcNow[ads.channel]/calibration_factor;
    }

private:
    /* Event handler */

    virtual void onDisconnectionComplete(const ble::DisconnectionCompleteEvent&) {
        _ble.gap().startAdvertising(ble::LEGACY_ADVERTISING_HANDLE);
    }

private:
    BLE &_ble;
    events::EventQueue &_event_queue;

    DigitalOut  _led1;
    ADS1256Service *_ads1256_service;
    Ads1256_t initialValue;
    
    int32_t ads_sum;

    UUID _ads1256_uuid;

    uint8_t _adv_buffer[ble::LEGACY_ADVERTISING_MAX_SIZE];
    ble::AdvertisingDataBuilder _adv_data_builder;
};

/** Schedule processing of events from the BLE middleware in the event queue. */
void schedule_ble_events(BLE::OnEventsToProcessCallbackContext *context) {
    event_queue.call(Callback<void()>(&context->ble, &BLE::processEvents));
}

int main()
{
    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(schedule_ble_events);

    ADS1256Demo demo(ble, event_queue);
    demo.start();

    return 0;
}

