/**********************************************************/
/*BitmapFile.h                                            */
/**********************************************************/

#include "mbed.h"

/* Class: BitmapFile
 * A parser for bitmap files.
*/
class BitmapFile
{
private:
	FILE			*m_pFile;
	char			*m_fileName;
	int				m_rowsize;
	
	enum headerType {	BITMAPCOREHEADER = 12,
						BITMAPCOREHEADER2 = 64,
						BITMAPINFOHEADER = 40,
						BITMAPV4HEADER = 108,
						BITMAPV5HEADER = 124};
						
	enum compressType {	BI_RGB,
						BI_RLE8,
						BI_RLE4,
						BI_BITFIELDS,
						BI_JPEG,
						BI_PNG};

	/* Struct: BMPHeader
	*
	* The BMP header of the bitmap is read into this.	
	*
	* b - the first byte of the header. Should equal 'B'.
	* m - the second byte of the header. Should equal 'M'.
	* filesize - the size of the whole file, in bytes.
	* reserved 1 and 2 - data specific to the applicaton which created the bitmap.
	* offset - the offset at which the actual bitmap begins
	*/
							
	__packed struct
	{
		char	b:8;
		char	m:8;
		int		filesize:32;
		int		reserved1:16;
		int		reserved2:16;
		int		offset:32;
		
	} BMPHeader;
	
	
	/* Struct: DIBHeader
	*
	* The DIB header of the bitmap is read into this.	
	*
	* headerlength - the length of the header. Should equal 40.
	* height - the height of the bitmap.
	* width - the width of the bitmap.
	* cplanes - the number of color planes. Should equal 1.
	* colordepth - the number of bits per pixel.
	* compression - the compression method used.
	* datasize - the size of the bitmap data, in bytes.
	*/

	int		m_headerlength;
	__packed struct
	{
		int		width:32;
		int		height:32;
		int		cplanes:16;
		int		colordepth:16;
		int		compression:32;
		int		datasize:32;
		int		hres:32;
		int		vres:32;
		int		numpalettecolors:32;
		int		importantcolors:32;		
	} DIBHeader;
	
						
public:
	/* Constructor: BitmapFile
	* Create the BitmapFile class, and call <Initialize>
	*
	* Parameters:
	*  fname - The path of the file to open.
	*/
	BitmapFile(char* fname);
	~BitmapFile();

	/* Function: Initialize
	* Parses the headers of the bitmap.
	*
	* Returns:
	*  Whether the bitmap is valid.
	*/
	bool	Initialize();	//parses the header

	/* Function: open
	* Opens the bitmap for reading, if not already open.
	*/
	void	open();

	/* Function: close
	* Closes the bitmap.
	*/
	void	close();

	/***BMP Header gets begin***/
	int		getFileSize();
	int		getReserved1();
	int		getReserved2();
	int		getOffset();
	
	/***DIB Header gets begin***/
	int		getHeaderType();
	int		getHeight();
	int		getWidth();
	int		getCPlanes();
	int		getColorDepth();
	int		getCompression();
	int		getDataSize();
	int		getHRes();
	int		getVRes();
	int		getNumPaletteColors();
	int		getImportantColors();
	/****DIB Header gets end****/
	
	/******Data gets begin******/
	/* Function: getPixel
	* Gets the color of a pixel
	*
	* Parameters:
	*  x - The x coordinate of the pixel.
	*  y - The y coordinate of the pixel.
	*  closefile - if specified, close the file after reading
	*
	* Returns:
	*  the color of the pixel, in hexadecimal.
	*/
	int		getPixel(int x, int y, bool closefile = true);
	/* Function: getRow
	* Gets the colors of a row
	*
	* Parameters:
	*  row - The number of the row..
	*  closefile - if specified, close the file after reading
	*
	* Returns:
	*  An array of the colors of the pixels, in hexadecimal.
	*/
	int		*getRow(int row, bool closefile = true);
	int		*getRowBW(int row, bool closefile = true);
	char	*getRowBitstream(int row, bool closefile = true);
	/*******Data gets end*******/
	int		getRowSize();
};