#include "BitmapFile.h"


BitmapFile::BitmapFile(char* fname) : m_pFile(NULL)
{
	m_fileName = fname;
	Initialize();
}

BitmapFile::~BitmapFile()
{
	delete[] m_fileName;
}

bool BitmapFile::Initialize()
{
	bool success = true;
	open();
	fread(&BMPHeader,sizeof(BMPHeader),1,m_pFile);
	success = (BMPHeader.b == 'B' && BMPHeader.m == 'M');
	
	fread(&m_headerlength,sizeof(m_headerlength),1,m_pFile);
	fread(&DIBHeader,m_headerlength,1,m_pFile);
	
	/*Debugging code
	
	Serial pc2(USBTX,USBRX);
	
	pc2.printf("\n\rFile = %s", m_fileName);
	
	pc2.printf("\n\rBMPHeader - Size = %d:\n\r",sizeof(BMPHeader));
	pc2.printf("\tbm:\t\t%c%c\n\r",BMPHeader.b,BMPHeader.m);
	pc2.printf("\tfilesize:\t%d\n\r",BMPHeader.filesize);
	pc2.printf("\treserved:\t%d,%d\n\r",BMPHeader.reserved1,BMPHeader.reserved2);
	pc2.printf("\toffset:\t\t%d\n\r",BMPHeader.offset);
	
	pc2.printf("\n\rDIBHeader - Size = %d:\n\r",sizeof(DIBHeader));
	pc2.printf("\theaderLength:\t%d\n\r",DIBHeader.headerLength);
	pc2.printf("\theight:\t\t\t%d\n\r",DIBHeader.height);
	pc2.printf("\twidth:\t\t%d\n\r",DIBHeader.width);
	pc2.printf("\tcplanes:\t\t%d\n\r",DIBHeader.cplanes);
	pc2.printf("\tcolordepth:\t\t%d\n\r",DIBHeader.colordepth);
	pc2.printf("\tcompression:\t%d\n\r",DIBHeader.compression);
	pc2.printf("\tdatasize:\t\t%d\n\r",DIBHeader.datasize);
	pc2.printf("\tvres:\t\t%d\n\r",DIBHeader.vres);
	pc2.printf("\thres:\t\t%d\n\r",DIBHeader.hres);
	
	
	*/
	
	m_rowsize = 4*((getColorDepth()*getWidth()+31)/32);
	
	close();
	return success;
}

void BitmapFile::open()
{
	if(m_pFile==NULL)
	{
		m_pFile = fopen(m_fileName, "r");
	}
}

void BitmapFile::close()
{
	if(m_pFile!=NULL)
	{
		fclose(m_pFile);
		m_pFile = NULL;
	}
}

/**********************************************************/
/*BMP Header Gets                                         */
/**********************************************************/

int BitmapFile::getFileSize()
{
	return BMPHeader.filesize;
}

int BitmapFile::getReserved1()
{
	return BMPHeader.reserved1;
}

int BitmapFile::getReserved2()
{
	return BMPHeader.reserved2;
}

int BitmapFile::getOffset()
{
	return BMPHeader.offset;
}

/**********************************************************/
/*DIB Header Gets                                         */
/**********************************************************/

int BitmapFile::getHeaderType()
{
	return m_headerlength;
}

int BitmapFile::getHeight()
{
	return DIBHeader.height;
}

int BitmapFile::getWidth()
{
	return DIBHeader.width;
}

int BitmapFile::getCPlanes()
{
	return DIBHeader.cplanes;
}

int BitmapFile::getColorDepth()
{
	return DIBHeader.colordepth;
}

int BitmapFile::getCompression()
{
	return DIBHeader.compression;
}

int BitmapFile::getDataSize()
{
	return DIBHeader.datasize;
}

int BitmapFile::getHRes()
{
	return DIBHeader.hres;
}

int BitmapFile::getVRes()
{
	return DIBHeader.vres;
}

int BitmapFile::getNumPaletteColors()
{
	return DIBHeader.numpalettecolors;
}

int BitmapFile::getImportantColors()
{
	return DIBHeader.importantcolors;
}

/**********************************************************/
/*Data Gets                                               */
/**********************************************************/

int BitmapFile::getRowSize()
{
	return m_rowsize;
}

int BitmapFile::getPixel(int row, int col, bool closefile)
{
	int color = -1;
	if(row>=0 && row < getHeight() && col>=0 && col< getWidth())
	{
		if(getColorDepth() == 24)
		{
			open();
			color = 0;	//make sure the last byte is 00
			
			int index = getOffset();
			index += col*3;
			index += row*4*ceil(getWidth()*3/4.0);
			fseek(m_pFile, index, SEEK_SET);
			
			fread (&color, 3,1,m_pFile);
			
			if(closefile)
			{
				close();
			}
		}
	}
	return color;
}

int *BitmapFile::getRow(int row, bool closefile)
{
	open();
	int *colors = new int[getWidth()];
	int index = getOffset() + m_rowsize*row;
	fseek(m_pFile, index, SEEK_SET);
	if(getColorDepth() == 24)
	{
		for(int i=0; i<getWidth(); i++)
		{
			fread(&colors[i],3,1,m_pFile);
		}
	}
	else if(getColorDepth() == 1)
	{
		char *temp = new char[m_rowsize];
		for(int i=0; i<m_rowsize; i++)
		{
			fread(&temp[i],sizeof(char),1,m_pFile);
		}
		for(int i=0; i<getWidth(); i++)
		{
			int byte = i / 8;
			int bit = i % 8;	
			colors[i] = ((temp[byte] << bit) & 0x80) ? 0xFFFFFF : 0x000000;
		}
		delete [] temp;
	}
	if(closefile)
	{
		close();
	}
	return colors;
}

int *BitmapFile::getRowBW(int row, bool closefile)
{
	open();
	int *colors = new int[getWidth()];
	int index = getOffset() + m_rowsize*row;
	fseek(m_pFile, index, SEEK_SET);
	if(getColorDepth() == 24)
	{
		for(int i=0; i<getWidth(); i++)
		{
			char temp[3];
			fread(temp,sizeof(char),3,m_pFile);
			int average = (temp[0]+temp[1]+temp[2])/3;
			colors[i] = average>128 ? 0xFFFFFF : 0x000000;
		}
	}
	else if(getColorDepth() == 1)
	{
		delete [] colors;
		colors = getRow(row, closefile);
	}
	if(closefile)
	{
		close();
	}
	return colors;
}

char *BitmapFile::getRowBitstream(int row, bool closefile)
{
	open();
	int bitsperrow = (getWidth()+7)/8;
	char *data = new char[bitsperrow];
	for(int i = 0; i<bitsperrow; i++)
	{
		data[i] = 0;
	}
	int index = getOffset() + m_rowsize*row;
	fseek(m_pFile, index, SEEK_SET);
	
	if(getColorDepth() == 24)
	{
		for(int i=0; i<getWidth(); i++)
		{
			char temp[3];
			fread(temp,sizeof(char),3,m_pFile);
			int average = (temp[0]+temp[1]+temp[2])/3;
			int val = average<128?0:1;
			data[i/8] |= (val*0x80) >> (i%8);
		}
	}
	else if(getColorDepth() == 1)
	{
		fread(data,sizeof(char),bitsperrow,m_pFile);
	}
	
	if(closefile)
	{
		close();
	}
	return data;
}