/*********************************************************************************************************************** 
This library is based upon the original TextLCD library. This library has been modified so it can be used with a 16x4 LCD
The traditional 'wait' function has been chnaged to use 16 bit Timer 1 rather than 32 bit Timer 0 which is what is set up 
usually in the mbed library. The reason for this change is so that the new wait function and this library could be used 
alongside a PulseWidthCapture program which needs to use the 32 bit timer. 

This program requires Library 'LCD_Wait' to work
Callum Ellor
**************************************************************************************************************************/

#include "TextLCD_16x4.h"
#include "LCD_Wait.h"
#include "mbed.h"


LCD_Wait lcd_wait;              // Use LCD_Wait library to replace standard wait function

TextLCD::TextLCD(PinName rs, PinName e, PinName d4, PinName d5,
                 PinName d6, PinName d7, LCDType type) : _rs(rs),
        _e(e), _d(d4, d5, d6, d7),
        _type(type) {

    _e  = 1;
    _rs = 0;            // command mode

    lcd_wait.Wait(0.015);        // Wait 15ms to ensure powered up

    // send "Display Settings" 3 times (Only top nibble of 0x30 as we've got 4-bit bus)
    for (int i=0; i<3; i++) {
        writeByte(0x3);
        lcd_wait.Wait(0.00164);  // this command takes 1.64ms, so wait for it
    }
    writeByte(0x2);     // 4-bit mode
    lcd_wait.Wait(0.000040f);    // most instructions take 40us

    writeCommand(0x28); // Function set 001 BW N F - -
    writeCommand(0x0C);
    writeCommand(0x6);  // Cursor Direction and Display Shift : 0000 01 CD S (CD 0-left, 1-right S(hift) 0-no, 1-yes
    cls();
}

void TextLCD::character(int column, int row, int c) {
    int a = address(column, row);
    writeCommand(a);
    writeData(c);
}

void TextLCD::cls() {
    writeCommand(0x01); // cls, and set cursor to 0
    lcd_wait.Wait(0.00164f);     // This command takes 1.64 ms
    locate(0, 0);
}


void TextLCD::locate(int column, int row) {
    _column = column;
    _row = row;
}

int TextLCD::_putc(int value) {
    if (value == '\n') {
        _column = 0;
        _row++;
        if (_row >= rows()) {
            _row = 0;
        }
    } else {
        character(_column, _row, value);
        _column++;
        if (_column >= columns()) {
            _column = 0;
            _row++;
            if (_row >= rows()) {
                _row = 0;
            }
        }
    }
    return value;
}

int TextLCD::_getc() {
    return -1;
}

void TextLCD::writeByte(int value) {
    _d = value >> 4;
    lcd_wait.Wait(0.000040f); // most instructions take 40us
    _e = 0;
    lcd_wait.Wait(0.000040f);
    _e = 1;
    _d = value >> 0;
    lcd_wait.Wait(0.000040f);
    _e = 0;
    lcd_wait.Wait(0.000040f);  // most instructions take 40us
    _e = 1;
}

void TextLCD::writeCommand(int command) {
    _rs = 0;
    writeByte(command);
}

void TextLCD::writeData(int data) {
    _rs = 1;
    writeByte(data);
}

int TextLCD::address(int column, int row) {
    switch (_type) {
        case LCD20x4:
            switch (row) {
                case 0:
                    return 0x80 + column;
                case 1:
                    return 0xc0 + column;
                case 2:
                    return 0x94 + column;
                case 3:
                    return 0xd4 + column;
            }
        case LCD16x4:                               // 16x4 LCD added
            switch (row) {
                case 0:
                    return 0x80 + column;
                case 1:
                    return 0xc0 + column;
                case 2:
                    return 0x90 + column;           //changed to work with 16x4
                case 3:
                    return 0xd0 + column;           //changed to work with 16x4
            }
        case LCD16x2B:
            return 0x80 + (row * 40) + column;
        case LCD16x2:
        case LCD20x2:
        default:
            return 0x80 + (row * 0x40) + column;
    }
}

int TextLCD::columns() {
    switch (_type) {
        case LCD20x4:
        case LCD20x2:
            return 20;
        case LCD16x2:
        case LCD16x2B:
        case LCD16x4:               //added to work with 16x4
        default:
            return 16;
    }
}

int TextLCD::rows() {
    switch (_type) {
        case LCD20x4:
        case LCD16x4:               //added to work with 16x4
            return 4;
        case LCD16x2:
        case LCD16x2B:
        case LCD20x2:
        default:
            return 2;
    }
}
