/**
 * @class QuadEnc
 * @date June 2nd, 2016
 * @author Weimen Li
 * @brief Header file for a class which encapsulates a quadrature encoder.
 * @remark The algorith used to read from the encoder may be found at:
 * http://makeatronics.blogspot.com/2013/02/efficiently-reading-quadrature-with.html
 */

#ifndef QUADENC_H_
#define QUADENC_H_
#include "mbed.h"

class QuadEnc {
public:
    /**
     * @brief Constructor for the QuadEnc object.
     * @param ChannelA The pin Channel A is connected to.
     * @param ChannelB The pin Channel B is connected to.
     * @param CPR The number of counts per revolution of the encoder.
     */
    QuadEnc(PinName ChannelA, PinName ChannelB, float CPR);
    virtual ~QuadEnc();

    /**
     * @brief Returns true when a new encoder count has been seen.
     * @returns Whether a new encoder count has been seen.
     * @remark Sets the hasNewCount flag to false after being called.
     */
    bool hasNewCount();
    /**
     * @brief Returns the current encoder count.
     * @returns The current encoder count.
     * @remark Declared to be inline for greater performance.
     */
    inline int32_t getCount() {
        // Generally, we would disable interrupts to avoid corruption here.
        // However, the returned value is the word size of the processor, so
        // It should be safe. Additionally, any corruption may only happen on the LSB.
        return count;
    }
    
    /**
     * @brief Returns the number of revolutions that have been counted.
     * @returns The number of revolutions that have been counted.
     */
    float getRevs();

    /**
     * @brief Resets the internal state variables of the encoder.
     * @returns void
     */
    void reset();
private:
     /// The current encoder count since the program started.
    volatile int32_t count;
     /// Whether the encoder has a new count.
    volatile bool bHasNewCount;
    /// The number of counts per revolution.
    const float CPR;
    
    /// Digital Input for Channel A.
    DigitalIn ChannelAIn;
    /// Digital Input for Channel B.
    DigitalIn ChannelBIn;
    
    /// Interrupt Object for Channel A.
    InterruptIn ChannelAInter;
    /// Interrupt Object for Channel B.
    InterruptIn ChannelBInter;

    /// Member Function ISR to be called when a change is detected on the channel pins.
    void QuadEncISR();

    /// The previous encoder count.
    int32_t prevCount;
    /// The previous number of revolutions.
    float prevRevs;
    /// The raw encoder reading.
    uint8_t enc_val;
};

#endif /* QUADENC_H_ */
