/**
 * @file HBridge.hpp
 * @author Weimen Li
 * @date June 4th, 2016
 * @brief HBridge Class Header
 * @class HBridge
 * @brief The HBridge class encapsulates a physical H-Bridge from which the direction
 * and PWM output can be controlled with separate pins.
 * Example Usage:
 * @code
// First, you must ensure that HBridge is included.
#include "HBridge.hpp"

// Create a new H-Bridge object -
// If many functions need access to it, do it out here, in a global scope.
// Otherwise, only create it in the function that you need it in.

// With a global scope:
HBridge HBridgeGlobal(PTA12, PTA13);

void someFunction() {
    // With a local scope: (May need to be static to persist across calls to the function.)
    static HBridge HBridgeLocal(PTA14, PTA15);
    HBridgeLocal.write(0.75);
}

int main() {

    for(;;) {
        // Write to an HBridge by calling its write method:
        HBridgeGlobal.write(0.50);
        someFunction();
    }
}
 * @endcode
 *
 */

#ifndef HBRIDGE_H_
#define HBRIDGE_H_
#include "mbed.h"

class HBridge {
public:
    /**
     * @brief Constructor for the H-bridge.
     * @param PWMPin The Pin that the motor PWM is attached to.
     * @param DirectionPin The Pin that the direction control is attached to.
     */
    HBridge(PinName PWMPin, PinName DirectionPin);
    virtual ~HBridge();
    /**
     * @brief Write an output to the H bridge.
     * @param percent a float between -1 and 1, where the magnitude indicates the
     * duty cycle, and the sign indicates the direction. Values exceeding the range are automatically
     * set to 1.
     */
    void write(float percent);
    /**
     * @brief Read the current output from the H-bridge.
     * @returns A float between -1 and 1, where the magnitude
     * indicates the duty cycle, and the sign indicates the direction.
     */
    float read();
private:
    PwmOut PWMOut;
    DigitalOut DirectionOut;
    bool outputHasChanged;
};

#endif /* HBRIDGE_H_ */
