/**
 * @file HC06Bluetooth.hpp
 * @date June 4th, 2016
 * @author Weimen Li
 * @class HC06Bluetooth
 * @brief This class creates an object representing the HC06 Bluetooth Module
 * The baud rate for the device is configured to be 115200.
 * @code
  // First, make sure to have included the hpp file:
  #include HC06Bluetooth.hpp
  // In the scope that you need it, create a new instance of the HC06Bluetooth object by calling:
  HC06Bluetooth HC06BluetoothObj(TXPin, RXPin, "My Cool Robot", someCallBackFunction);
  // The "someCallBackFunction" is a function you write that is called when a new line has been read.
  // For instance, one simple callback is to simply echo (reprint to output) the received line:

  void BTEchoCallback(const char* receivedLine) {
      HC06BluetoothObj.print(receivedLine);
  }

  // A more sophisticated callback would be one that understands the received line contains commands,
  // and parses them:
// Callback function to read a command from input
// Include the <sstream> library, since that is what we use to process information.
#include <sstream>
#include <string>
// Variables to read information into.
volatile float kConst;
volatile float iConst;
volatile float dConst;
void BTCommandCallback(const char* receivedString) {
    stringstream stringStream(receivedString);
    // The received string has the form "newDeviceName kConst iConst dConst"
    // Necessary to use these temporary variables since stringStream cannot read into
    // "volatile float", but it can read into "float".
    float kConstTemp;
    float iConstTemp;
    float dConstTemp;
    stringStream >> kConstTemp;
    stringStream >> iConstTemp;
    stringStream >> dConstTemp;
    kConst = kConstTemp;
    iConst = iConstTemp;
    dConst = dConstTemp;
    // Echo the received commands:
    char output[256];
    sprintf(output, "Received Command: kConst = %f, iConst = %f, dConst = %f", kConst, iConst, dConst);
    bluetoothObj.print(output);
}

}
 * @endcode
 */

//TODO: Complete this example.


#ifndef HC06BLUETOOTH_H_
#define HC06BLUETOOTH_H_
#include "mbed.h"
#include <string>

    const int dataBufferSize = 256;

class HC06Bluetooth {
public: // Public methods.
    /**
     * @brief Constructor for the HC06_Bluetooth class.
     * @param TX The pin that the TX line is attached to.
     * @param RX The pin that the RX line is attached to.
     * @param deviceName The name that you want your system to be identified as when you connect to it i.e. "Weimen's MAE433Robot"
     * @param password A 4-digit numeric PIN that you want your device to connect with. It defaults to "1234".
     * @param lineCallbackFunc The callback function that will be called once a newline character is encountered on the receiving data.
     * The callback function takes as an argument a string containing the line that has been read.
     * @remark The callback function is run within within an interrupt service routine, so it should be written to be safe for ISRs.
     * @param charCallbackFunc A function that will be called once a new character has been read. It should return void and take as an argument
     * the character that has been read.
     */
    HC06Bluetooth(PinName TX = PTD3, PinName RX = PTD2, void (*lineCallbackFunc) (const char* readString) = NULL, void (*charCallbackFunc) (char readChar) = NULL);
    virtual ~HC06Bluetooth();
    /**
     * @brief Run the setup routine to configure the device name and the device pin.
     * @remark: You only need to run this once during the entire time you have the module, since the device name and PIN are saved
     * by the module itself! However, you may run the function again if you would like to change the deviceName or PIN.
     * @param deviceName The name of the device, as a string. It must only contain alphanumeric characters - no spaces allowed.
     * @param PIN The device PIN.
     */
    void runSetup(std::string deviceName, std::string PIN = "1234");
    /**
     * @brief Print information in buffer to the output.
     * @param buffer A null-terminated buffer containing the data you want to send.
     */
    void print(const char *buffer);
    /**
     * @brief Print a character to output.
     * @param char The character you want to print to output.
     */
    void print(char c);

private:
    RawSerial btSerialObj;
    void receiveByteISR();
    /// Pointer to a callback function the client provides when a line is received.
    void (*lineCallbackFunc) (const char*);
    /// Pointer to a callback function the client provides when a character is received.
    void (*charCallbackFunc) (char);
    char dataReceivedBuffer[dataBufferSize];
    int32_t dataReceivedBufferPos;
    char dataReceivedBufferCopy[dataBufferSize];
};

#endif /* HC06BLUETOOTH_H_ */
