/**
 * @file FXOS8700CQ.hpp
 * @author Weimen Li
 * @date June 5th, 2016
 * @class FXOS8700CQ
 * @brief This class represents the FXOS8700CQ sensor. Only accelerometer reading is currently supported.
 * @remark WARNING: I2C Reading is unreliable when built outside of the mbed compiler. If you're compiling this outside
 * of the mBed compiler, ensure that the accelerometer is reading reliably. My testing shows that the readings often glitch.
 */

#ifndef FXOS8700CQ_HPP_
#define FXOS8700CQ_HPP_
#include <mbed.h>

class FXOS8700CQ {
public:
	/// Possible accelerometer sensitivity settings.
	enum AccelerometerSensitivity {TWO, FOUR, EIGHT};
	/**
	 * @brief Constructor for the FXOS8700CQ Accelerometer/Magnetometer. Pin assignments have default values for the K22F board.
	 * @param SDA The SDA Pin.
	 * @param SCL the SCL Pin.
	 * @param INT1 The pin that the INT1 line is connected to.
	 * @param INT2 The pin that the INT2 line is connect to.
	 * @param setting (optional) The maximum measurement acceleration in g's. May be TWO, FOUR, or EIGHT for
	 * +/- 2/4/8 g's. A smaller maximum measurement allows for greater sensitivity.
	 */
	FXOS8700CQ(PinName SDA = PTB3, PinName SCL = PTB2, PinName INT1 = PTD0, PinName INT2 = PTD1, AccelerometerSensitivity setting = TWO);
	virtual ~FXOS8700CQ();
	/**
	 * @brief Read data from the accelerometer. Takes pointers to floats that store these variables.
	 * Example code:
	 * @code
	 * float xAccel;
	 * float yAccel;
	 * float zAccel;
	 * ...
	 * readAccelerometer(&xAccel, &yAccel, &zAccel);
	 * ...
	 * @endcode
	 */
	void readAccelerometer(float *xAccel, float *yAccel, float *zAccel);
	/**
	 * @brief Set offset compensation values that are subtracted from the readings to zero them. 
	 */
	 void setOffset(float x, float y, float z) {
	 	xOffset = x;
	 	yOffset = y;
	 	zOffset = z;
	 }
private:
	/// Member function to handle data-ready interrupt.
	void dataReadyISR(void);
private:
	/// I2C Object to handle bus communications.
	I2C I2CObj;
	/// Interrupt object for a data-ready signal.
	InterruptIn dataReadyInt;
	/// The accelerometer sensitivity setting specified by the caller.
	const AccelerometerSensitivity accelSensitivitySetting;
	/// The conversion constant to convert the read int16_t type data to a float.
	const float accelInt2Float;
	/// The latest x Acceleration value in g's.
	float lastxAccel;
	/// The latest y Acceleration value in g's.
	float lastyAccel;
	/// The latest z Acceleration values in g's.
	float lastzAccel;
	/// The x offset
	float xOffset;
	/// The y offset
	float yOffset;
	/// The z offset
	float zOffset;
};

#endif /* FXOS8700CQ_HPP_ */

