/**
 * @file HC06Bluetooth.hpp
 * @date June 4th, 2016
 * @author Weimen Li
 * @class HC06Bluetooth
 * @brief This class creates an object representing the HC06 Bluetooth Module
 * The baud rate for the device is configured to be 115200.
 */


#ifndef HC06BLUETOOTH_H_
#define HC06BLUETOOTH_H_
#include "mbed.h"
#include "rtos.h"
#include <string>
#include <queue>

const uint32_t SIG_BT_BYTE = (1 << 2);
// WARNING: DO NOT CHANGE THESE VALUES, AS THEY ARE USED TO INDEX INTO AN ARRAY FOR IMPLEMENTATION.
enum Baudrate {B1200, B2400, B4800, B9600, B19200, B38400, B57600, B115200, B230400, B460800, B921600, B1382400, END};

class HC06Bluetooth {
public: // Public methods.
    /**
     * @brief Constructor for the HC06_Bluetooth class.
     * @param TX The pin that the TX line is attached to.
     * @param RX The pin that the RX line is attached to.
     * @param deviceName The name that you want your system to be identified as when you connect to it i.e. "Weimen's MAE433Robot"
     * @param password A 4-digit numeric PIN that you want your device to connect with. It defaults to "1234".
     * @param lineCallbackFunc The callback function that will be called once a newline character is encountered on the receiving data.
     * The callback function takes as an argument a string containing the line that has been read.
     * @remark The callback function is run within within an interrupt service routine, so it should be written to be safe for ISRs.
     * @param charCallbackFunc A function that will be called once a new character has been read. It should return void and take as an argument
     * the character that has been read.
     * @remark The callback function is run within within an interrupt service routine, so it should be written to be safe for ISRs.
     */
    HC06Bluetooth(PinName TX, PinName RX, Baudrate baudrate = B115200, void (*lineCallbackFunc) (const char* readString, size_t strlen) = NULL, void (*charCallbackFunc) (char readChar) = NULL);
    virtual ~HC06Bluetooth();
    /**
     * @brief Run the setup routine to configure the device name and the device pin.
     * @remark: You only need to run this once during the entire time you have the module, since the device name and PIN are saved
     * by the module itself! However, you may run the function again if you would like to change the deviceName or PIN.
     * @param deviceName The name of the device, as a string. It must only contain alphanumeric characters - no spaces allowed.
     * @param PIN The device PIN.
     */
    void runSetup(std::string deviceName, std::string PIN = "1234");
    /**
     * @brief Print information in buffer to the output.
     * @param buffer A null-terminated buffer containing the data you want to send.
     */
    void print(const char *buffer);
    void print(std::string s) {
       print(s.c_str());
    }
    /**
     * @brief Print information in buffer to the output followed by a newline
     * @param buffer A null-terminated buffer containing the data you want to send.
     */
    void println(const char *buffer);
    void println(std::string s) {
       println(s.c_str());
    }
    /**
     * @brief Print a character to output.
     * @param char The character you want to print to output.
     */
    void print(char c);

private:
    RawSerial btSerialObj;
    Baudrate baudrate;
    void receiveByteISR();
    void receiveByteThread();
    Thread receiveByteThreadObj;
    /// Pointer to a callback function the client provides when a line is received.
    void (*lineCallbackFunc) (const char*, size_t strlen);
    /// Pointer to a callback function the client provides when a character is received.
    void (*charCallbackFunc) (char);
    Queue<char, 64> dataReceivedBuffer;
    std::queue<char> dataReceivedBufferCopy;
    std::vector<char> dataReceivedToClient;
};

#endif /* HC06BLUETOOTH_H_ */
