#ifndef GAMEENGINE_H
#define GAMEENGINE_H

#include "Border.h"
#include "Food.h"
#include "Gamepad.h"
#include "mbed.h"
#include "N5110.h"
#include "Snake.h"

/** GameEngine Class
* @brief Merges sub-class functions, writes object states, and processes object-specific collisions
* @author EL16RF
* @date Mar, 2018
*/

class GameEngine
{

public:
    /** Constructor */
    GameEngine();
    /** Deconstructor */
    ~GameEngine();

    // functions
    /** Initialises the game engine */
    void init();
    
    /** Reads inputs from the gamepad
    * @param &gamepad (Gamepad) - the input source */
    void readInput(Gamepad &gamepad);

    /** Updates the game engine
    * @param &gamepad (Gamepad) - the input source
    * @param border_type (int) - the snake control type (0-1) */
    void update(Gamepad &gamepad, N5110 &lcd, int border_type);
    
    /** Draws game objects into the LCD buffer
    * @param border_type (int) - the border type (0-1)
    * @param &lcd (N5110) - the buffer write target */
    void draw(int border_type, N5110 &lcd);
    
    /** Writes object-specific LCD states for collision detection
    * @param &lcd (N5110) - the buffer write target
    * @param border_type (int) - the border type (0-1) */
    void writeObjectStates(N5110 &lcd, int border_type);

private:
    // objects
    Direction _dir;
    ABXYEvent _abxy;
    Snake _snake;
    Food _food;
    Border _border;
    
    // functions
    void writeLCDState(int state[WIDTH][HEIGHT], N5110 &lcd);
    void checkCollision(Gamepad &gamepad, N5110 &lcd);
    bool selfCollision(int x, int y, Gamepad &gamepad);
    bool foodCollision(int x, int y, Gamepad &gamepad);
    bool borderCollision(int x, int y, Gamepad &gamepad);
    void gameOver(Gamepad &gamepad, N5110 &lcd);

    // variables
    int _snake_x;
    int _snake_y;
    int _snake_head_state[WIDTH][HEIGHT];
    int _snake_tail_state[WIDTH][HEIGHT];
    int _food_state[WIDTH][HEIGHT];
    int _border_state[WIDTH][HEIGHT];
    int _score;
};

#endif