#include "GameEngine.h"

// constructor
GameEngine::GameEngine()
{

}

// deconstructor
GameEngine::~GameEngine()
{

}

void GameEngine::init()
{

    // call initialisation functions from object classes
    _snake.init(40, 32);
    _food.init();
    _score = 0;
}

void GameEngine::readInput(Gamepad &gamepad)
{

    // get snake control inputs from the gamepad
    _dir = gamepad.get_direction();
    _abxy = gamepad.get_abxy();
}

void GameEngine::update(Gamepad &gamepad, N5110 &lcd, int control_type)
{

    // get snake head x and y position
    LCDVector snake_position = _snake.getPosition();
    _snake_x = snake_position.x;
    _snake_y = snake_position.y;
    // printf("%d %d\n", _snake_x, _snake_y);

    // check for collisions
    checkCollision(gamepad, lcd);

    // update the snake
    _snake.update(control_type, _dir, _abxy);
}

void GameEngine::draw(int border_type, N5110 &lcd)
{

    // call draw functions from object classes
    _border.draw(border_type, lcd);
    _food.draw(lcd);
    _snake.draw(lcd);
}

void GameEngine::writeObjectStates(N5110 &lcd, int border_type)
{

    // clear lcd buffer
    lcd.clear();
    // draw the snake head into the buffer
    _snake.drawHead(lcd);
    // printf("snake head state written to LCD buffer\n");
    // write the state of the LCD to the snake head state array
    writeLCDState(_snake_head_state, lcd);

    // clear lcd buffer
    lcd.clear();
    // draw the snake tail into the buffer
    _snake.drawTail(lcd);
    // printf("snake tail state written to LCD buffer\n");
    // write the state of the LCD to the snake tail state array
    writeLCDState(_snake_tail_state, lcd);

    // clear lcd buffer
    lcd.clear();
    // draw the food into the buffer
    _food.draw(lcd);
    // printf("food state written to LCD buffer\n");
    // write the state of the LCD to the food state array
    writeLCDState(_food_state, lcd);

    // clear lcd buffer
    lcd.clear();
    // draw the specified border type into the buffer
    _border.draw(border_type, lcd);
    // printf("border state written to LCD buffer\n");
    // write the state of the LCD to the border state array
    writeLCDState(_border_state, lcd);
}

void GameEngine::writeLCDState(int state[WIDTH][HEIGHT], N5110 &lcd)
{

    // for each pixel on the screen
    // y - top to bottom
    for (int i = 0; i < HEIGHT; i++) {

        // x - left to right
        for (int j = 0; j < WIDTH; j++) {

            // copy lcd state to state array
            state[j][i] = lcd.getPixel(j, i);
            // printf("%d", state[j][i]);
        }

        // printf("\n");
    }

    // printf("lcd pixel state written to state array\n");
}

void GameEngine::checkCollision(Gamepad &gamepad, N5110 &lcd)
{

    // set conflict flag
    bool conflict_flag = false;

    // for each pixel on the screen
    // y - top to bottom
    for (int i = 0; i < HEIGHT; i++) {

        // x - left to right
        for (int j = 0; j < WIDTH; j++) {

            // check food collision
            if (foodCollision(j, i, gamepad)) {

                // update conflict flag
                conflict_flag = true;
                // break the secondary for loop
                break;
            }

            // check border collision
            if (borderCollision(j, i, gamepad)) {

                // enter game over
                gameOver(gamepad, lcd);
            }

            // check self collision
            if (selfCollision(j, i, gamepad)) {

                // enter game over
                gameOver(gamepad, lcd);
            }
        }

        // if the conflict flag = true
        if (conflict_flag) {

            // break the inital for loop
            break;
        }
    }
}

bool GameEngine::selfCollision(int x, int y, Gamepad &gamepad)
{

    // for the snake tail length (excluding the head)
    for (int i = 1; i < _snake.getLength(); i++) {

        // if the snake head state and the snake tail state both = 1 at the same position
        if (_snake_head_state[x][y] == 1 && _snake_tail_state[x][y] == 1) {

            // printf("self collision detected (render vector x: %dpx y: %dpx)\n", _snake_x, _snake_y);

            // return self collision as true
            return true;
        }
    }

    // return self collision as false
    return false;
}

bool GameEngine::foodCollision(int x, int y, Gamepad &gamepad)
{

    // if the snake head state and food state arrays both = 1 at the same position
    if (_snake_head_state[x][y] == 1 && _food_state[x][y] == 1) {

        // printf("food collision detected (render vector x: %dpx y: %dpx)\n", _snake_x, _snake_y);
        // increment the score
        _score++;
        // increment the snake length
        _snake.incrementLength();
        // printf("snake length updated: %d\n", _snake.getLength());
        // update the food
        _food.update();
        // play 440Hz for 100ms
        gamepad.tone(440, 0.1f);

        // return food collision as true
        return true;
    }

    // return food collision as false
    return false;
}

bool GameEngine::borderCollision(int x, int y, Gamepad &gamepad)
{

    // if the snake head state and border state arrays both = 1 at the same position
    if (_snake_head_state[x][y] == 1 && _border_state[x][y] == 1) {

        // printf("border collision detected (render vector x: %dpx y: %dpx)\n", _snake_x, _snake_y);

        // return border collision as true
        return true;
    }

    // return border collision as false
    return false;
}

void GameEngine::gameOver(Gamepad &gamepad, N5110 &lcd)
{

    // printf("game over\n");
    
    // play 1.76kHz for 1s
    gamepad.tone(1760, 1.0f);
    // light red leds
    gamepad.led(1, 1.0f);
    gamepad.led(4, 1.0f);
    
    // draw game over
    lcd.clear();
    lcd.drawRect(0, 0, WIDTH, HEIGHT, FILL_BLACK);
    lcd.printString("GAME OVER", 18, 1);
    lcd.refresh();
    
    wait(1.0f);
    
    // draw game over and score screen    
    char score[14];
    sprintf(score, "%.3d", _score);
    lcd.printString("Score", 30, 3);
    lcd.printString(score, 36, 4);
    lcd.refresh();
    
    wait(4.0f);
    
    // system reset
    // https://os.mbed.com/questions/5754/Is-it-possible-to-do-a-software-reset-on/
    NVIC_SystemReset();
}