/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */
#include "mbed.h"
#include "stdint.h"
#include "stdio.h"
#include <algorithm>
#include <string.h>
#include <stdlib.h>

#define MAX_COUNTER 5 /* This is in seconds*/

/*Machine state describes whether the machine is ON or OFF
Port_Param can be     @   mode_Select_Pot
                            water_mode_Pot
                            Load_size_Port
*/

typedef struct Washing_Machine {
  int Machine_State;
  int Port_Param;
  int count_down;
  float Temperature;
  char Pot1_mode[20];
  char Pot2_water[20];
  char Pot3_Load[20];
  char Temp[20];
} Washing_MachineTypedef;

typedef enum Pot_Select {
  mode_Select_Pot = 1,
  water_mode_Pot,
  Load_size_Port,
} Pot_Select;

typedef enum M_Status { OFF = 0, ON, RUN } M_Status;

Washing_MachineTypedef Machine = {0};
/*               "n"  "p"  "d" "h"  "c"  "s"   "l"*/
int HexLabel[]={0x15,0x67,0x3D,0x17,0x4E,0x5B,0x0E};
int PotValue1, PotValue2, PotValue3;
BusOut SegDis(PC_7,PA_12,PB_1,PB_15,PB_14,PB_12,PA_11);//change PC_7  to PB_11 
Ticker TIMER;                /*Declare a Timer for Timing Duration*/
DigitalOut Buzzer(PA_15);    /*Buzzer connected to PA_15*/
DigitalOut Red_Led(PC_0);    /*RED LED to signify machine in stop mode*/
DigitalOut Green_Led(PC_1);  /*Green LED to signify machine in running mode*/
DigitalOut Yellow_Led(PC_2); /*Green LED to signify machine in Setup Mode mode*/
InterruptIn POWER_ON(PC_10); /*Pin to power On the machine connecte to pin PC_10*/
InterruptIn POWER_OFF(PC_11); /*Pin to power OFF the machine connecte to pin PC_11*/
InterruptIn Start_Sequence( PA_0); /*Pin to start the machine sequence connected to pin PC_11*/
AnalogIn Pot1_mode_Select(PA_5); /*To select @ Normal, Permanent or Delicate*/
AnalogIn Pot2_water_mode(PA_6);  /*To select @ Warm water or Cool water*/
AnalogIn Pot3_Load_size(PA_7); /* To select the size of the load, which can small or large*/
AnalogIn LM35(PC_3); /*Analogue Pin for reading LM35*/

void TimerCount_Down() 
{ 
    Machine.count_down--; 
}
void Application() 
{
  if (Machine.Port_Param > 3) 
  {
    Machine.Port_Param = 1;
  }
  if (Machine.Port_Param == mode_Select_Pot) 
  {
    PotValue1 = Pot1_mode_Select.read_u16();
    if (PotValue1 < 21845) 
    {
      sprintf(Machine.Pot1_mode, "%s", "NORMAL");
       SegDis.write(HexLabel[0]);

    }
    if (PotValue1 >= 21845 && PotValue1 < 43690) 
    {
      sprintf(Machine.Pot1_mode, "%s", "PERMANENT");
      SegDis.write(HexLabel[1]);

    }
    if (PotValue1 >= 43690) 
    {
      sprintf(Machine.Pot1_mode, "%s", "DELICATE");
      SegDis.write(HexLabel[2]);
 
    }
  }
  if (Machine.Port_Param == water_mode_Pot) 
  {
    PotValue2 = Pot2_water_mode.read_u16();
    if (PotValue2 < 32768) 
    {
      sprintf(Machine.Pot2_water, "%s", "WARM");
      SegDis.write(HexLabel[3]);

    }
    if (PotValue2 >= 21845) 
    {
      sprintf(Machine.Pot2_water, "%s", "COLD");
      SegDis.write(HexLabel[4]);

    }
  }
  if (Machine.Port_Param == Load_size_Port) 
  {
    PotValue3 = Pot3_Load_size.read_u16();
    if (PotValue3 < 32768) 
    {
      sprintf(Machine.Pot3_Load, "%s", "SMALL");
      SegDis.write(HexLabel[5]);

    }
    if (PotValue3 >= 32768) 
    {
      sprintf(Machine.Pot3_Load, "%s", "LARGE");
      SegDis.write(HexLabel[6]);

    }
  }
}
void Power_On_function() 
{
  
  Machine.Machine_State = ON;
  Machine.Port_Param++;
}
void Power_Off_Machine() 
{ 
    Machine.Machine_State = OFF; 
}
void Run_Machine() 
{
  Machine.count_down = MAX_COUNTER;      /*Initialize the counter to MAX seconds*/
  TIMER.attach(&TimerCount_Down,1.0);
  Machine.Machine_State = RUN;
}
void LM35_Temperature() 
{ 
    Machine.Temperature = ((LM35.read() * 3276.0) / 10.0);
    sprintf(Machine.Temp, "%f", Machine.Temperature);
    printf("TEMP= %.2f\n\r", Machine.Temperature);
}
int main() 
{
  POWER_ON.mode(PullUp);                /*Activete pull up for power on button*/
  POWER_OFF.mode(PullUp);               /*Activete pull up for power off button*/
  Start_Sequence.mode(PullUp);        /*Activete pull up for run sequence button*/
  POWER_ON.fall(&Power_On_function);     /*Power ON the system*/
  POWER_OFF.fall(&Power_Off_Machine);    /*Power OFF the system*/
  Start_Sequence.fall(&Run_Machine);    /*start Sequence of the system*/
  while (true) 
  {

    switch (Machine.Machine_State) 
    {
    case OFF:
      SegDis.write(0x00);
      Red_Led = 1;
      Yellow_Led = 0;
      Green_Led = 0;
      break;
    case ON:
      Buzzer = 0;
      Yellow_Led = 1;
      Red_Led = 0;
      Green_Led = 0;
      Application();
      LM35_Temperature();
      //printf("Pot1_mode=%s,Pot2_water=%s,Pot3_Load=%s\n\r",Machine.Pot1_mode, Machine.Pot2_water, Machine.Pot3_Load);     
      break;
    case RUN:
      Red_Led = 0;
      Green_Led = 1;
      Yellow_Led = 0;
      if (Machine.count_down == 0) 
      {
        TIMER.detach();
        Buzzer = 1;
        Machine.Machine_State = ON;
        SegDis.write(0x00);
        wait_us(1000000);
         Buzzer = 0;
      } 
      else 
      {
        printf("%d s\n\r", Machine.count_down);
      }

      break;
    }
    wait_us(100000);
  }
}
