#include "jsmn_utils.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

// TEST COMMENT TO COMMIT

void
jsmn_error(int err_code)
{
    switch(err_code) {
        case JSMN_ERROR_INVAL:
            printf("Invalid JSON object\r\n");
            break;
        case JSMN_ERROR_NOMEM:
            printf("Not enough tokens slots allocated\r\n");
            break;
        case JSMN_ERROR_PART:
            printf("JSON object incomplete\r\n");
            break;
        default:
            printf("Unknown error\r\n");
    }
}

//+----------------------------------------------------------------------------+
//                      GETTING VALUE FROM JSON OBJECT
//+----------------------------------------------------------------------------+

jsmntok_t* jsmn_get_token(char* json_string, jsmntok_t* token, char* target)
{
    size_t key_num = token->size;
    size_t current = 0;
    token++;

    while(current < key_num) {

        size_t length = token->end - token->start;
        char key[length + 1];
        key[length] = '\0';
        memcpy(key,json_string + token->start,length);


        if(strcmp(key, target) == 0) {
            return token;
        } else {
            token += token->size + 1;
        }
        current++;
    }

    return NULL;
}

size_t
jsmn_get_string_length(char* json_string, jsmntok_t* token, char *target, int idx)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;
    if(tok->type == JSMN_ARRAY) {
        tok += idx + 1;
        return (tok->end - tok->start);
    } else {
        return (tok->end - tok->start);
    }
}

void
jsmn_get_string(char* json_string, jsmntok_t* token, char* target, int idx, char* dest)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;
    if(tok->type == JSMN_ARRAY) {
        tok += idx + 1;
    }
    memcpy(dest, json_string + tok->start, tok->end - tok->start);
}

int
jsmn_get_integer(char* json_string, jsmntok_t* token, char* target, int idx)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;
    if(tok->type == JSMN_ARRAY) {
        tok += idx + 1;
    }
    return (int) strtol(json_string + tok->start, (char**)NULL,  10);
}

unsigned long
jsmn_get_ulong(char* json_string, jsmntok_t* token, char* target, int idx)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;
    if(tok->type == JSMN_ARRAY) {
        tok += idx + 1;
    }
    return strtoul(json_string + tok->start, (char**)NULL,  10);
}

bool
jsmn_get_boolean(char* json_string, jsmntok_t* token, char* target, int idx)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;
    if(tok->type == JSMN_ARRAY) {
        tok += idx + 1;
    }
    return json_string[tok->start] == 't' ? true : false;
}

void jsmn_get_integer_array(char* json_string, jsmntok_t* token, char* target, int* destination)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;

    if(!(tok->type == JSMN_ARRAY)) {
        printf("\"%s\" is not an array\r\n", target);
    } else {
        size_t length = tok->size;
        for(int i = 0; i < length; i++) {
            tok++;
            destination[i] = (int) strtol(json_string + tok->start, (char**)NULL, 10);
        }
    }
}

void jsmn_get_ulong_array(char* json_string, jsmntok_t* token, char* target, unsigned long* destination)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;

    if(!(tok->type == JSMN_ARRAY)) {
        printf("\"%s\" is not an array\r\n", target);
    } else {
        size_t length = tok->size;
        for(int i = 0; i < length; i++) {
            tok++;
            destination[i] = strtoul(json_string + tok->start, (char**)NULL, 10);
        }
    }
}

void jsmn_get_boolean_array(char* json_string, jsmntok_t* token, char* target, bool* destination)
{
    jsmntok_t* tok = jsmn_get_token(json_string, token, target);
    tok++;

    if(!(tok->type == JSMN_ARRAY)) {
        printf("\"%s\" is not an array\r\n", target);
    } else {
        size_t length = tok->size;
        for(int i = 0; i < length; i++) {
            tok++;
            destination[i] = (json_string[tok->start] == 't') ? true : false;
        }
    }
}



//+----------------------------------------------------------------------------+
//                           PRINTING JSON OBJECTS
//+----------------------------------------------------------------------------+

jsmntok_t*
jsmn_print_token(char *json_string, jsmntok_t *token)
{
    int  start  = token->start;
    int  end    = token->end;
    int  length = token->size;
    char tmp    = json_string[start];

    switch(token->type) {
        case JSMN_OBJECT: {
            char obj_string[end-start + 1];
            memcpy(obj_string, json_string + start, end-start);
            obj_string[end-start] = '\0';

            jsmn_print_object(obj_string);
            token++;
        }
        return token;

        case JSMN_ARRAY:
            printf("[\r\n");
            for(int i = 0; i < length; i++) {
                printf("    ");
                token++;
                jsmn_print_token(json_string, token);
            }
            printf("]\r\n");
            token++;
            return token;

        case JSMN_STRING:
            printf("%.*s\r\n", end - start, json_string + start);
            token++;
            return token;

        case JSMN_PRIMITIVE:
            switch(tmp) {
                case 't':
                    printf("true\r\n");
                    break;
                case 'f':
                    printf("false\r\n");
                    break;
                case 'n':
                    printf("null\r\n");
                    break;
                default: {
                    long tmp_long = strtol(json_string + start, NULL, 10);
                    if (tmp_long == 2147483647) {
                        unsigned long tmp_ulong = (unsigned long) strtoul(json_string + start, NULL, 10);
                        if (tmp_ulong == 4294967295) {
                            long long int tmp_lli = strtoll(json_string + start, NULL, 10);
                            printf("%lld\r\n", tmp_lli);
                        } else {
                            printf("%lu\r\n", tmp_ulong);
                        }
                    } else {
                        printf("%li\r\n", tmp_long);
                    }
                }
            }

            token++;
            return token;

        default:
            printf("Unknown type\r\n");
            token++;
            return token;
    }
}

void
jsmn_print_object(char *json_string)
{
    jsmn_parser parser;
    jsmn_init(&parser);

    int toks = jsmn_parse(&parser, json_string, strlen(json_string), NULL, 0);
    jsmntok_t tokens[toks];

    jsmn_init(&parser);
    jsmn_parse(&parser, json_string, strlen(json_string), tokens, toks);

    jsmntok_t *token;
    token = tokens;

    printf("{\r\n");

    size_t key_num = token->size;
    token++;
    size_t current = 0;
    while(current < key_num) {
        // Print key
        printf("%.*s: ", token->end - token->start, json_string + token->start);
        token++;

        // Print value
        token = jsmn_print_token(json_string, token);

        current++;
    }

    printf("}\r\n");
}
