#ifndef CONVERTER_C
#define CONVERTER_C

#include "./converter.h"

#include "./common.h"
#include "./types.h"

static const trit_t trits_mapping[27][3] = {
    {-1, -1, -1}, {0, -1, -1}, {1, -1, -1}, {-1, 0, -1}, {0, 0, -1}, {1, 0, -1},
    {-1, 1, -1},  {0, 1, -1},  {1, 1, -1},  {-1, -1, 0}, {0, -1, 0}, {1, -1, 0},
    {-1, 0, 0},   {0, 0, 0},   {1, 0, 0},   {-1, 1, 0},  {0, 1, 0},  {1, 1, 0},
    {-1, -1, 1},  {0, -1, 1},  {1, -1, 1},  {-1, 0, 1},  {0, 0, 1},  {1, 0, 1},
    {-1, 1, 1},   {0, 1, 1},   {1, 1, 1}
};


static const char tryte_to_char_mapping[] = "NOPQRSTUVWXYZ9ABCDEFGHIJKLM";

void chars_to_trytes(char *chars_in, tryte_t *trytes_out, int len)
{
    for (int i = 0; i < len; i++) {
        if (*(chars_in + i) == '9') {
            *(trytes_out + i) = 0;
        } else if (*(chars_in + i) >= 'N') {
            *(trytes_out + i) = (int8_t)(*(chars_in + i)) - 64 - 27;
        } else {
            *(trytes_out + i) = (int8_t)(*(chars_in + i)) - 64;
        }
    }
}

void trytes_to_chars(tryte_t trytes_in[],
                     char    chars_out[],
                     int     len)
{
    for (int i = 0; i < len; i++) {
        chars_out[i] = tryte_to_char_mapping[trytes_in[i] + 13];
    }
}

void trytes_to_trits(tryte_t trytes_in[],
                     trit_t  trits_out[],
                     int     tryte_len)
{
    for (int i = 0; i < tryte_len; i++) {
        int8_t idx = (int8_t)trytes_in[i] + 13;
        trits_out[i * 3 + 0] = trits_mapping[idx][0];
        trits_out[i * 3 + 1] = trits_mapping[idx][1];
        trits_out[i * 3 + 2] = trits_mapping[idx][2];
    }
}

void trits_to_trytes(trit_t  trits_in[],
                     tryte_t trytes_out[],
                     int     trit_len)
{
    if (trit_len % 3 != 0) {
        THROW(INVALID_LENGTH_TRIT_TRYTE_CONV);
    }
    int tryte_len = trit_len / 3;

    for (int i = 0; i < tryte_len; i++) {
        trytes_out[i] = trits_in[i * 3 + 0] + trits_in[i * 3 + 1] * 3 +
                        trits_in[i * 3 + 2] * 9;
    }
}

void
trits_to_chars(trit_t *trits, char *chars, int length)
{
    tryte_t trytes[length/3];
    trits_to_trytes(trits,trytes,length);
    trytes_to_chars(trytes,chars,length/3);
}

void
chars_to_trits(char *chars, trit_t *trits, int length)
{
    tryte_t trytes[length];
    chars_to_trytes(chars,trytes,length);
    trytes_to_trits(trytes,trits,length);
}

#endif // CONVERTER_C