#include "mbed.h"
#include "stm32l4xx_hal_flash.h"
#include "stm32l4xx_hal_flash_ex.h"
#include "flash_programming.h"

#define FLASH_PAGE_PROGRAM 1

uint32_t DATA_32 = 0x12345222;
uint64_t DATA_64 = 0x1234522212345222;

uint32_t StartPage = 0, EndPage = 0;
uint32_t Address = 0;
uint32_t PageError = 0;

__IO FlashReturnStatus MemoryProgramStatus = PASSED;

FLASH_EraseInitTypeDef EraseInitStruct;
FLASH_OBProgramInitTypeDef OptionsBytesStruct, OptionsBytesStruct2;

Serial serial(USBTX, USBRX);

DigitalOut led3(LED1);

void main() {
    serial.baud(115200);
    printf("PROGRAM STARTS\r\n");
    led3 = 0;
    
  /* Initialize test status */
  MemoryProgramStatus = PASSED;
  
  /* Unlock the Flash to enable the flash control register access *************/ 
  HAL_FLASH_Unlock();

  /* Clear OPTVERR bit set on virgin samples */
  __HAL_FLASH_CLEAR_FLAG(FLASH_FLAG_OPTVERR); 

  /* Unlock the Options Bytes *************************************************/
  HAL_FLASH_OB_Unlock();

  /* Get the number of the start and end pages */
  StartPage = GetPage(FLASH_USER_START_ADDR);
  EndPage   = GetPage(FLASH_USER_END_ADDR);

  OptionsBytesStruct.WRPArea  = OB_WRPAREA_BANK1_AREAA;
  OptionsBytesStruct2.WRPArea = OB_WRPAREA_BANK1_AREAB;

  /* Get pages write protection status ****************************************/
  HAL_FLASHEx_OBGetConfig(&OptionsBytesStruct);
  HAL_FLASHEx_OBGetConfig(&OptionsBytesStruct2);

  /* Configure write protected pages */
  if (OptionsBytesStruct.OptionType == OPTIONBYTE_WRP)
  {
    if(HAL_FLASHEx_OBProgram(&OptionsBytesStruct) != HAL_OK)
    {
      /* Error occurred while options bytes programming. **********************/
      while (1)
      {
        /* Make LED3 blink (100ms on, 2s off) to indicate error in operation */
        led3 = 1;
        wait(100/1000);
        led3 = 0;
        wait(2000/1000);
      }
    }
  }

  if (OptionsBytesStruct2.OptionType == OPTIONBYTE_WRP)
  {
    if(HAL_FLASHEx_OBProgram(&OptionsBytesStruct2) != HAL_OK)
    {
      /* Error occurred while options bytes programming. **********************/
      while (1)
      {
        /* Make LED3 blink (100ms on, 2s off) to indicate error in operation */
        led3 = 1;
        wait(100/1000);
        led3 = 0;
        wait(2000/1000);
      }
    }
  }

  /* Generate System Reset to load the new option byte values ***************/
  if ((OptionsBytesStruct.OptionType == OPTIONBYTE_WRP) || (OptionsBytesStruct2.OptionType == OPTIONBYTE_WRP))
  {
    HAL_FLASH_OB_Launch();
  }

  /* Lock the Options Bytes *************************************************/
  HAL_FLASH_OB_Lock();

#ifdef FLASH_PAGE_PROGRAM  
printf("Test flash step 2\r\n");
  /* The selected pages are write protected *******************************/
  if (((OptionsBytesStruct.WRPStartOffset  <= StartPage) && (OptionsBytesStruct.WRPEndOffset  >= EndPage)) || 
      ((OptionsBytesStruct2.WRPStartOffset <= StartPage) && (OptionsBytesStruct2.WRPEndOffset >= EndPage)))
  {
    printf("Step 2: 1\r\n");
    /* The desired pages are write protected */ 
    /* Check that it is not allowed to write in this page */
    Address = FLASH_USER_START_ADDR; 
    if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, Address, DATA_64) != HAL_OK)
    {
      /* Error returned during programmation. */
      /* Check that WRPERR flag is well set */
      if ((HAL_FLASH_GetError() & HAL_FLASH_ERROR_WRP) != 0) 
      {
        MemoryProgramStatus = FAILED;
      }
      else
      {
        /* Another error occurred.
           User can add here some code to deal with this error */
        while (1)
        {
          /* Make LED3 blink (100ms on, 2s off) to indicate error in Write operation */
          led3 = 1;
          wait(100/1000);
          led3 = 0;
          wait(2000/1000);
        }
      }
    }
    else
    {
      /* Write operation is successful. Should not occur
         User can add here some code to deal with this error */
      while (1)
      {
        /* Make LED3 blink (100ms on, 2s off) to indicate error in Write operation */
        led3 = 1;
        wait(100/1000);
        led3 = 0;
        wait(2000/1000);
      }
    }
  }
  else
  {
    printf("Step 2: 2\r\n");
    /* The desired pages are not write protected */ 
    /* Fill EraseInit structure************************************************/

    EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
    EraseInitStruct.Banks       = FLASH_BANK_1;
    EraseInitStruct.Page        = StartPage;
    EraseInitStruct.NbPages     = EndPage - StartPage + 1;
    
    printf("Step 2: 2: 1\r\n");
    if (HAL_FLASHEx_Erase(&EraseInitStruct, &PageError) != HAL_OK)
    {
      printf("Step 2: 2: 2\r\n");
      /* 
        Error occurred while page erase. 
        User can add here some code to deal with this error. 
        PageError will contain the faulty page and then to know the code error on this page,
        user can call function 'HAL_FLASH_GetError()'
      */
      while (1)
      {
        /* Make LED3 blink (100ms on, 2s off) to indicate error in Erase operation */
        led3 = 1;
        wait(100/1000);
        led3 = 0;
        wait(2000/1000);
      }
    }
    printf("Step 2: 2: 3\r\n");
    /* FLASH Word program of DATA_32 at addresses defined by FLASH_USER_START_ADDR and FLASH_USER_END_ADDR */
    Address = FLASH_USER_START_ADDR;
    while (Address < FLASH_USER_END_ADDR)
    {
      printf("Step 2: 2: 4\r\n");
      if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, Address, DATA_64) == HAL_OK)
      {
        Address = Address + 8;       
      }
      else
      {
        /* Error occurred while writing data in Flash memory. 
           User can add here some code to deal with this error */
        while (1)
        {
          /* Make LED3 blink (100ms on, 2s off) to indicate error in Write operation */
          led3 = 1;
          wait(100/1000);
          led3 = 0;
          wait(2000/1000);
        }
      }
    }

    /* Check the correctness of written data */
    Address = FLASH_USER_START_ADDR;

    while (Address < FLASH_USER_END_ADDR)
    {
      if((*(__IO uint32_t*) Address) != DATA_32)
      {
        MemoryProgramStatus = FAILED;
      }
      Address += 4;
    }
  }
#endif /* FLASH_PAGE_PROGRAM */

  /* Lock the Flash to disable the flash control register access (recommended
     to protect the FLASH memory against possible unwanted operation) *********/
  HAL_FLASH_Lock();

  /*Check if there is an issue to program data*/
  if (MemoryProgramStatus == PASSED)
  {
    /* No error detected. Switch on LED3*/
    printf("Write OK\r\n");
    led3 = 1;
  }
  else
  {
    printf("Write failed\r\n");
    /* Error detected. LED3 will blink with 1s period */
    while (1)
    {
      led3 = 1;
      wait(1000/1000);
      led3 = 0;
      wait(1000/1000);
    }
  }
}