#include "flash_programming.h"
#include "stm32l4xx_hal_flash.h"

uint32_t PageError = 0;

FLASH_EraseInitTypeDef EraseInitStruct;

uint32_t FP_GetPage(uint32_t Addr) {
  uint32_t page = 0;
  
  if (Addr < (FLASH_BASE + FLASH_BANK_SIZE))
  {
    /* Bank 1 */
    page = (Addr - FLASH_BASE) / FLASH_PAGE_SIZE;
  }
  else
  {
    /* Bank 2 */
    page = (Addr - (FLASH_BASE + FLASH_BANK_SIZE)) / FLASH_PAGE_SIZE;
  }
  
  return page;
}

int FP_ClearFlags() {
    __HAL_FLASH_CLEAR_FLAG(FLASH_FLAG_EOP | FLASH_FLAG_PGAERR | FLASH_FLAG_OPTVERR | FLASH_FLAG_PGSERR);
    if((__HAL_FLASH_GET_FLAG(FLASH_FLAG_OPERR))  || (__HAL_FLASH_GET_FLAG(FLASH_FLAG_PROGERR)) || 
       (__HAL_FLASH_GET_FLAG(FLASH_FLAG_WRPERR)) || (__HAL_FLASH_GET_FLAG(FLASH_FLAG_PGAERR))  || 
       (__HAL_FLASH_GET_FLAG(FLASH_FLAG_SIZERR)) || (__HAL_FLASH_GET_FLAG(FLASH_FLAG_PGSERR))) {
        printf("Clear flag error\r\n");
        return FAILED;
    }    
    return PASSED;
}

uint32_t FP_ReadValue(uint32_t Addr) {
    uint32_t ReturnValue = *(__IO uint32_t*)Addr;
    return ReturnValue;
}

int FP_WriteRelayStates(uint8_t RelayState1, uint8_t RelayState2, uint8_t RelayState3) {
    uint8_t  CurrentPage = FP_GetPage(RELAY_BASE_ADDRESS);
    uint32_t CurrentAddress = RELAY1_ADDRESS;
    
    EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
    EraseInitStruct.Banks       = FLASH_BANK_1;
    EraseInitStruct.Page        = CurrentPage;
    EraseInitStruct.NbPages     = 1;      
    
    if (FP_ClearFlags() != PASSED) {
        return FAILED;
    }
    HAL_FLASH_Unlock();  
    
    if ((FP_ReadValue(RELAY1_ADDRESS) == RelayState1) &&
        (FP_ReadValue(RELAY2_ADDRESS) == RelayState2) &&
        (FP_ReadValue(RELAY3_ADDRESS) == RelayState3)) {
        printf("Relay values don't change, no need to write\r\n");
        HAL_FLASH_Lock();
        return PASSED;
    }
    if (HAL_FLASHEx_Erase(&EraseInitStruct, &PageError) != HAL_OK) {
        printf("Erase error, error num %d\r\n", HAL_FLASH_GetError());
    }  
    while (CurrentAddress < (RELAY3_ADDRESS + STEP_ADDRESS)) {
        switch (CurrentAddress) {
            case (RELAY1_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, RelayState1) == HAL_OK) {
                                        printf("Write Relay 1 State OK\r\n");
                                   }
                                   else {
                                        printf("Write Relay 1 State failed, error num %d\r\n", HAL_FLASH_GetError());
                                   }
            break;
            case (RELAY2_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, RelayState2) == HAL_OK) {
                                        printf("Write Relay 2 State OK\r\n");
                                   }
                                   else {
                                        printf("Write Relay 2 State failed, error num %d\r\n", HAL_FLASH_GetError());
                                   }
            break;
            case (RELAY3_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, RelayState3) == HAL_OK) {
                                        printf("Write Relay 3 State OK\r\n");
                                   }
                                   else {
                                        printf("Write Relay 3 State failed, error num %d\r\n", HAL_FLASH_GetError());
                                   }
            break;
            default: break;   
        }
        CurrentAddress = CurrentAddress + STEP_ADDRESS;
    }

    CurrentAddress = RELAY1_ADDRESS;
    while (CurrentAddress < (RELAY3_ADDRESS + STEP_ADDRESS)) {
        switch (CurrentAddress) {
            case (RELAY1_ADDRESS): if (FP_ReadValue(CurrentAddress) == RelayState1) {
                                    printf("Read back Relay 1 State: %d\r\n", FP_ReadValue(CurrentAddress));
                                 }
                                 else {
                                    printf("Write Relay 1 State failed, wrong read back value\r\n"); 
                                    HAL_FLASH_Lock();
                                    return FAILED;
                                 }
            break;
            case (RELAY2_ADDRESS): if (FP_ReadValue(CurrentAddress) == RelayState2) {
                                        printf("Read back Relay 2 State: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write Relay 2 State failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            case (RELAY3_ADDRESS): if (FP_ReadValue(CurrentAddress) == RelayState3) {
                                        printf("Read back Relay 3 State: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write Relay 3 State failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            default: break;   
        }
        CurrentAddress = CurrentAddress + STEP_ADDRESS;
    }
    HAL_FLASH_Lock();
    return PASSED;
}

int FP_WriteConfigValues(uint8_t Mode ,uint8_t OxyThres, uint8_t TempThres, uint32_t UploadPeriod) {
    uint8_t  CurrentPage    = FP_GetPage(CONF_BASE_ADDRESS);    
    uint32_t CurrentAddress = MODE_ADDRESS;
    EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
    EraseInitStruct.Banks       = FLASH_BANK_1;
    EraseInitStruct.Page        = CurrentPage;
    EraseInitStruct.NbPages     = 1;     

    if (FP_ClearFlags() != PASSED) {
        return FAILED;
    }
    HAL_FLASH_Unlock();  
    
    if ((FP_ReadValue(MODE_ADDRESS) == Mode) &&
        (FP_ReadValue(OXY_THRES_ADDRESS) == OxyThres) &&
        (FP_ReadValue(TEMP_THRES_ADDRESS) == TempThres) &&
        (FP_ReadValue(UPLOAD_PERIOD_ADDRESS) == UploadPeriod)) {
        printf("Configuration values don't change, no need to write\r\n");
        HAL_FLASH_Lock();
        return PASSED;
    }
    if (HAL_FLASHEx_Erase(&EraseInitStruct, &PageError) != HAL_OK) {
        printf("Erase error, error num %d\r\n", HAL_FLASH_GetError());
    }  
    while (CurrentAddress < (UPLOAD_PERIOD_ADDRESS + STEP_ADDRESS)) {
        switch (CurrentAddress) {
            case (MODE_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, Mode) == HAL_OK) {
                                    printf("Write Mode OK\r\n");
                                }
                                else {
                                    printf("Write Mode failed, error num %d\r\n", HAL_FLASH_GetError());
                                }
            break;
            case (OXY_THRES_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, OxyThres) == HAL_OK) {
                                        printf("Write OxyThres OK\r\n");
                                    }
                                    else {
                                        printf("Write OxyThres failed, error num %d\r\n", HAL_FLASH_GetError());
                                    }
            break;
            case (TEMP_THRES_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, TempThres) == HAL_OK) {
                                        printf("Write TempThres OK\r\n");
                                    }
                                    else {
                                        printf("Write TempThres failed, error num %d\r\n", HAL_FLASH_GetError());
                                    }
            break;
            case (UPLOAD_PERIOD_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, UploadPeriod) == HAL_OK) {
                                        printf("Write UploadPeriod OK\r\n");
                                    }
                                    else {
                                        printf("Write UploadPeriod failed, error num %d\r\n", HAL_FLASH_GetError());
                                    }
            break;
            default: break;   
        }
        CurrentAddress = CurrentAddress + STEP_ADDRESS;
    }
    
    CurrentAddress = MODE_ADDRESS;
    while (CurrentAddress < (UPLOAD_PERIOD_ADDRESS + STEP_ADDRESS)) {
        switch (CurrentAddress) {
            case (MODE_ADDRESS): if (FP_ReadValue(CurrentAddress) == Mode) {
                                    printf("Read back Mode: %d\r\n", FP_ReadValue(CurrentAddress));
                                 }
                                 else {
                                    printf("Write Mode failed, wrong read back value\r\n"); 
                                    HAL_FLASH_Lock();
                                    return FAILED;
                                 }
            break;
            case (OXY_THRES_ADDRESS): if (FP_ReadValue(CurrentAddress) == OxyThres) {
                                        printf("Read back OxyThres: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write OxyThres failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            case (TEMP_THRES_ADDRESS): if (FP_ReadValue(CurrentAddress) == TempThres) {
                                        printf("Read back TempThres: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write TempThres failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            case (UPLOAD_PERIOD_ADDRESS): if (FP_ReadValue(CurrentAddress) == UploadPeriod) {
                                        printf("Read back UploadPeriod: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write UploadPeriod failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            default: break;   
        }
        CurrentAddress = CurrentAddress + STEP_ADDRESS;
    }
    HAL_FLASH_Lock();
    return PASSED;
}

int FP_WriteConfigValues(uint32_t SaturationDoVoltage) {
    uint8_t  CurrentPage    = FP_GetPage(SAT_DO_VOLT_ADDRESS);    
    uint32_t CurrentAddress = SAT_DO_VOLT_ADDRESS;
    EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
    EraseInitStruct.Banks       = FLASH_BANK_1;
    EraseInitStruct.Page        = CurrentPage;
    EraseInitStruct.NbPages     = 1;     

    if (FP_ClearFlags() != PASSED) {
        return FAILED;
    }
    HAL_FLASH_Unlock();     

    if ((FP_ReadValue(SAT_DO_VOLT_ADDRESS) == SaturationDoVoltage)) {
        printf("Configuration values doesn't change, no need to write\r\n");
        HAL_FLASH_Lock();
        return PASSED;
    }
    if (HAL_FLASHEx_Erase(&EraseInitStruct, &PageError) != HAL_OK) {
        printf("Erase error, error num %d\r\n", HAL_FLASH_GetError());
    }  
    if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, SaturationDoVoltage) == HAL_OK) {
        printf("Write SaturationDoVoltage OK\r\n");
    }
    else {
        printf("Write SaturationDoVoltage failed, error num %d\r\n", HAL_FLASH_GetError());
    }
    if (FP_ReadValue(CurrentAddress) == SaturationDoVoltage) {
        printf("Read back SaturationDoVoltage: %d\r\n", FP_ReadValue(CurrentAddress));  
    }
    else {
        printf("Write SaturationDoVoltage failed, wrong read back value\r\n");
        HAL_FLASH_Lock();
        return FAILED;
    }
    HAL_FLASH_Lock();
    return PASSED;
}

int FP_SetAlarmValues(uint32_t WriteTimeValue, uint8_t SetRelayState1, uint8_t SetRelayState2) {
    uint8_t  CurrentPage = FP_GetPage(ALRM_BASE_ADDRESS);
    uint32_t CurrentAddress = ALARM_TIME_ADDRESS;
    
    EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
    EraseInitStruct.Banks       = FLASH_BANK_1;
    EraseInitStruct.Page        = CurrentPage;
    EraseInitStruct.NbPages     = 1;      
    
    if (FP_ClearFlags() != PASSED) {
        return FAILED;
    }
    HAL_FLASH_Unlock();  
    
    if ((FP_ReadValue(ALARM_TIME_ADDRESS)  == WriteTimeValue) &&
        (FP_ReadValue(SET_RELAY_1_ADDRESS) == SetRelayState1) &&
        (FP_ReadValue(SET_RELAY_2_ADDRESS) == SetRelayState2)) {
        printf("Relay values don't change, no need to write\r\n");
        HAL_FLASH_Lock();
        return PASSED;
    }
    if (HAL_FLASHEx_Erase(&EraseInitStruct, &PageError) != HAL_OK) {
        printf("Erase error, error num %d\r\n", HAL_FLASH_GetError());
    }  
    while (CurrentAddress < (SET_RELAY_2_ADDRESS + STEP_ADDRESS)) {
        switch (CurrentAddress) {
            case (ALARM_TIME_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, WriteTimeValue) == HAL_OK) {
                                        printf("Write WriteTimeValue OK\r\n");
                                   }
                                   else {
                                        printf("Write WriteTimeValue failed, error num %d\r\n", HAL_FLASH_GetError());
                                   }
            break;
            case (SET_RELAY_1_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, SetRelayState1) == HAL_OK) {
                                        printf("Write SetRelayState1 OK\r\n");
                                   }
                                   else {
                                        printf("Write SetRelayState1 failed, error num %d\r\n", HAL_FLASH_GetError());
                                   }
            break;
            case (SET_RELAY_2_ADDRESS): if (HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, CurrentAddress, SetRelayState2) == HAL_OK) {
                                        printf("Write SetRelayState2 OK\r\n");
                                   }
                                   else {
                                        printf("Write SetRelayState2 failed, error num %d\r\n", HAL_FLASH_GetError());
                                   }
            break;
            default: break;   
        }
        CurrentAddress = CurrentAddress + STEP_ADDRESS;
    }

    CurrentAddress = ALARM_TIME_ADDRESS;
    while (CurrentAddress < (SET_RELAY_2_ADDRESS + STEP_ADDRESS)) {
        switch (CurrentAddress) {
            case (ALARM_TIME_ADDRESS): if (FP_ReadValue(CurrentAddress) == WriteTimeValue) {
                                    printf("Read back WriteTimeValue: %d\r\n", FP_ReadValue(CurrentAddress));
                                 }
                                 else {
                                    printf("Write WriteTimeValue failed, wrong read back value\r\n"); 
                                    HAL_FLASH_Lock();
                                    return FAILED;
                                 }
            break;
            case (SET_RELAY_1_ADDRESS): if (FP_ReadValue(CurrentAddress) == SetRelayState1) {
                                        printf("Read back SetRelayState1: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write SetRelayState1 failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            case (SET_RELAY_2_ADDRESS): if (FP_ReadValue(CurrentAddress) == SetRelayState2) {
                                        printf("Read back SetRelayState2: %d\r\n", FP_ReadValue(CurrentAddress));  
                                    }
                                    else {
                                        printf("Write SetRelayState2 failed, wrong read back value\r\n");
                                        HAL_FLASH_Lock();
                                        return FAILED;
                                    }
            break;
            default: break;   
        }
        CurrentAddress = CurrentAddress + STEP_ADDRESS;
    }
    HAL_FLASH_Lock();
    return PASSED;    
}