#ifndef __BLE_BIKE_SERVICE_H__
#define __BLE_BIKE_SERVICE_H__

#include "ble/BLE.h"

/**
  * @class bikeService
  * @brief This is a customized service for a device controlling the bike via BLE
  */
class bikeService {   
public:
    const static uint16_t BIKE_SERVICE_UUID                   = 0xA580;
    const static uint16_t SWITCH_CONTROL_UUID                 = 0xA581;   
    const static uint16_t TEMPERATURE_MEASURE_UUID            = 0xA582;   
    const static uint16_t HUMIDITY_MEASURE_UUID               = 0xA583;   
    const static uint16_t AIR_QUALITY_MEASURE_UUID            = 0xA584;   
    
     
/** 
  * @param[in] _ble BLE object for the underlying controller.
  * @param[in] currentTemperature The temperature measured from the sensor
  * @param[in] currentHumidity    The humidity measured from the sensor
  */

    bikeService(BLE &_ble, bool initialSwitchState, int8_t currentTemperature, uint8_t currentHumidity /*, uint16_t airQuality */) :
        ble(_ble), 
        switchState(SWITCH_CONTROL_UUID, &initialSwitchState),
        temperatureCharacteristic(GattCharacteristic::UUID_TEMPERATURE_CHAR, &currentTemperature, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        humidityCharacteristic(GattCharacteristic::UUID_HUMIDITY_CHAR, &currentHumidity, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        airQualityCharacteristic(AIR_QUALITY_MEASURE_UUID, &airQuality, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)        
    {
            /* Create and add the service */    
            GattCharacteristic *charTable[] = {&switchState, &temperatureCharacteristic, &humidityCharacteristic /*, &airQualityCharacteristic */};
            GattService         bikeService(BIKE_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
            ble.gattServer().addService(bikeService);
    }

    GattAttribute::Handle_t getValueHandle() const {
        return switchState.getValueHandle();
    }

    void updateTemperatureCharacteristic(int8_t newTemperature) {
        currentTemperature = newTemperature; 
        ble.gattServer().write(temperatureCharacteristic.getValueHandle(), (uint8_t *) &currentTemperature, sizeof(int8_t));
    }
    
    void updateAirQualityCharacteristic(uint8_t newHumidity) {
        currentHumidity = newHumidity; 
        ble.gattServer().write(humidityCharacteristic.getValueHandle(), (uint8_t *) &currentHumidity, sizeof(uint8_t));
    }
    
    void updateTemperatureCharacteristic(uint16_t newAirQuality) {
        airQuality = newAirQuality;
        ble.gattServer().write(airQualityCharacteristic.getValueHandle(), (uint8_t *) &airQuality, sizeof(uint16_t));
    }
    
    void updateAllCharacteristics(int8_t newTemperature, uint8_t newHumidity /*, uint16_t newAirQuality */) {
        currentTemperature = newTemperature;
        currentHumidity = newHumidity;
/*        airQuality = newAirQuality; */
        
        ble.gattServer().write(temperatureCharacteristic.getValueHandle(), (uint8_t *) &currentTemperature, sizeof(int8_t));
        ble.gattServer().write(humidityCharacteristic.getValueHandle(), (uint8_t *) &currentHumidity, sizeof(uint8_t));
/*        ble.gattServer().write(airQualityCharacteristic.getValueHandle(), (uint8_t *) &airQuality, sizeof(uint16_t));  */
    }
private:
    int8_t currentTemperature;
    uint8_t currentHumidity;
    uint16_t airQuality;

    BLE                                  &ble;
    ReadWriteGattCharacteristic<bool>     switchState;
    ReadOnlyGattCharacteristic<int8_t>    temperatureCharacteristic;
    ReadOnlyGattCharacteristic<uint8_t>   humidityCharacteristic;
    ReadOnlyGattCharacteristic<uint16_t>  airQualityCharacteristic;
};

#endif /* __BLE_BIKE_SERVICE_H__ */
