/** 
  * This is the code for "BLE Device for motorbike". The device is attached on any bike at will.
  * User can control 2 switches and these switches can control anything that user wants ie: turn on
  * the bike, turn on the alarm system of the bike, turn on the light...
  * Temperature sensor is also included in the device. User can view the temperature when he/she gets
  * near the bike.
  
  * Revision:
  * v0_1: only one switch is used.
  * v0_5: another switch is added.
  * v0_7: thermometer is added. User can view thermomether data on the phone. 
  * v0_8: humidity and air quality are added. User can view measured data on the phone. 
  * v1_0: a real DHT22 library is added.
  */

/* ======================== INCLUDES ========================= */
#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "DHT22.h"
#include "bike_service.h"

/* ======================== DEFINES ========================== */

/* ======================= VARIABLES ========================= */
/* GLOBAL VARIABLES */
int8_t currentTemperature = 19;
uint8_t currentHumidity    = 35;
uint16_t currentAirQuality = 550;

/* PRIVATE VARIABLES */
const static char     DEVICE_NAME[] = "WAVE ALPHA 8214";
static const uint16_t uuid16_list[] = {bikeService::BIKE_SERVICE_UUID};

/* STRUCTS/CLASSESS */
DigitalOut         switch1(LED2, 0);
DigitalOut         switch2(LED3, 0);
static EventQueue  eventQueue(EVENTS_EVENT_SIZE * 20);
bikeService       *bikeServicePtr;

/* ================== FUNCTION PROTOTYPES ==================== */

/* ==================== FUNCTION DETAILS ===================== */
/* A connection should be re-enabled after the device and the device
 * disconnected
 */
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params) {
    (void) params;
    BLE::Instance().gap().startAdvertising();
}

void main_event(void) {
    /* Any code can be added here */
    currentTemperature++;
    currentHumidity += 7;
    currentAirQuality += 129;
    bikeServicePtr->updateAllCharacteristics(currentTemperature, currentHumidity);
}

/**
 * @brief     This callback allows the bikeService to receive updates to the switchState Characteristic.
 * @param[in] params
 */
void receiveDataCallback(const GattWriteCallbackParams *params) {
    if ((params->handle == bikeServicePtr->getValueHandle()) && (params->len == 1)) {
        uint8_t receiveData = *(params->data);
        
        switch (receiveData) {
            case 0x00: switch1 = 0; break;
            case 0x01: switch1 = 1; break;
            case 0x10: switch2 = 0; break;
            case 0x11: switch2 = 1; break;
            
            default: break;   
        }
    }
}

/**  
  * @brief This function is called when the ble initialization process has failled
  */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Error handle here */
}

/** 
  * @brief Callback triggered when the ble initialization process has finished
  */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);
    ble.gattServer().onDataWritten(receiveDataCallback);

    bool initialSwitchState = false;
    bikeServicePtr = new bikeService(ble, initialSwitchState, currentTemperature, currentHumidity);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms. */
    ble.gap().startAdvertising();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

/* MAIN FUNCTION */
int main() {
    eventQueue.call_every(500, main_event);

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    eventQueue.dispatch_forever();

    return 0;
}
