/* Atmel Inertial One IMU Library
 * Copyright (c) 2012 Daniel Kouba
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
 #ifndef IMU_AIO_H        //IMU Atmel Inertial One
 #define IMU_AIO_H
 
 #include "mbed.h"
 
 /* Defines */
 
 //7 bit I2C Addresses, shifted left to allow for read/write bit as LSB
 
 #define GYRO_ADR  0x68 << 1
 #define ACC_ADR   0x38 << 1
 #define COMP_ADR  0x0C << 1
 
 // ITG3200 Gyro Registers
 
 #define GYRO_WHO_AM_I_REG         0x00
 #define GYRO_SAMPLE_DIV_REG       0x15
 #define GYRO_DLPF_REG             0x16
 #define GYRO_INT_CFG_REG          0x17
 #define GYRO_INT_STATUS_REG       0x1A
 #define GYRO_TEMP_H_REG           0x1B
 #define GYRO_TEMP_L_REG           0x1C
 #define GYRO_XOUT_H_REG           0x1D
 #define GYRO_XOUT_L_REG           0x1E
 #define GYRO_YOUT_H_REG           0x1F
 #define GYRO_YOUT_L_REG           0x20
 #define GYRO_ZOUT_H_REG           0x21
 #define GYRO_ZOUT_L_REG           0x22
 
 // Gyro LPF bandwidths
 
 #define BW_256HZ       0x00        //8kHz sample rate
 #define BW_188HZ       0x01        //1kHz sample rate for 188Hz and below
 #define BW_98HZ        0x02
 #define BW_42HZ        0x03
 #define BW_20HZ        0x04
 #define BW_10HZ        0x05
 #define BW_5HZ         0x06
 
 // Gyro Initial FS_SEL Register Config
 
 #define FS_SEL_INIT    0x03 << 3
  
 // BMA150 Accelerometer Registers
 
 #define ACC_XOUT_L_REG            0x02     //Acceleration data registers
 #define ACC_XOUT_H_REG            0x03
 #define ACC_YOUT_L_REG            0x04
 #define ACC_YOUT_H_REG            0x05
 #define ACC_ZOUT_L_REG            0x06
 #define ACC_ZOUT_H_REG            0x07
 #define ACC_TEMP_REG              0x08     //Temperature register, 0.5 deg. C / LSB
 #define ACC_CTRL_REG              0x0A     //Control registers
 #define ACC_OPER_REG              0x14     //Operational registers
 #define ACC_INT_REG               0x15     //Interrupt registers
 
 // Accelerometer LPF bandwidths
 
 #define BW_25HZ        0x00
 #define BW_50HZ        0x01
 #define BW_100HZ       0x02
 #define BW_190HZ       0x03
 #define BW_375HZ       0x04
 #define BW_750HZ       0x05
 #define BW_1500HZ      0x06
 
 // Accelerometer Range values
 
 #define RANGE_2G       0x00
 #define RANGE_4G       0x01
 #define RANGE_8G       0x02
   
 // AK8973 Compass Registers

 
 /** Atmel Inertial One IMU Control Class
 * 
 * Includes control routines for:
 * - ITG-3200 3-axis, 16 bit gyroscope
 * - BMA-150 3-axis, 10 bit accelerometer
 * - AK8975 3-axis, 16 bit magnetometer
 *
 * Datasheets:
 *
 * http://www.atmel.com/dyn/resources/prod_documents/doc8354.pdf
 * 
 * http://invensense.com/mems/gyro/documents/PS-ITG-3200A.pdf
 *
 * http://www.bosch-sensortec.com/content/language1/downloads/BMA150_DataSheet_Rev.1.5_30May2008.pdf
 * 
 * http://pdf1.alldatasheet.com/datasheet-pdf/view/219477/AKM/AK8973/+Q18W89VYpLawLCDwv+/datasheet.pdf
 * 
 */
 class IMU {
 
 public:
    struct data3d
    {
        int x, y, z;
    } gyro_data, acc_data, mag_data;
 
 public:    
 
    /** Creates IMU object and initializes all three chips
    *
    * Gyro: FS_SEL register is set to 0x03, as required by datasheet
    *
    * Accelerometer:
    *
    * Compass:
    *
    * @param sda pin for I2C sda signal
    * @param scl pin for I2C scl signal
    */
    IMU(PinName sda, PinName scl);
    
    /* Gyro Methods */
    
    /** Gets current ADC data from all axes of gyro (uses burst read mode)
    * 
    * @return data3d struct of X, Y, and Z axis gyro measurements (signed 16 bits)
    */
    data3d getGyroData(void);
    
    /** Sets digital LPF bandwidth for all gyro channels - Not working currently
    * 
    * @param bw Filter Bandwidth (use defined bandwidths)
    */
    void setGyroLPF(char bw);
    
    /* Accelerometer Methods */
    
    /** Gets current X acceleration
    *
    * @return Raw X acceleration, 10 bits signed
    */
    int accX(void);

    /** Gets current Y acceleration
    *
    * @return Raw Y acceleration, 10 bits signed
    */
    int accY(void);

    /** Gets current Z acceleration
    *
    * @return Raw Z acceleration, 10 bits signed
    */
    int accZ(void);
    
    /** Gets current acceleration on all axes
    *
    * @return data3d struct of acceleration data, signed ints
    */
    data3d getAccData(void);
    
    /** Sets digital LPF filter bandwidth
    *
    * @param bw Digital LPF filter bandwidth
    */
    void setAccLPF(char bw);

    /** Sets accelerometer measurement range (+/-2, 4, or 8g's)
    *
    * @param range Range of g measurements
    */    
    void setAccRange(char range);
    
    /** Updates all image registers with data stored in EEPROM
    *
    */    
    void accUpdateImage(void);

    /** Set EEPROM write enable
    *
    * @param we Write enable value
    */    
    void accEEWriteEn(bool we);
    
    /* Compass Methods */
    
    
 
 private:
 
    I2C _i2c;       //I2C object constructor
 
 };
 
 #endif