
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
#include "mbed.h"
#define __DEBUG
#include "mydbg.h"
#include "usb_mem.h"
#include "string.h" //For memcpy, memmove, memset
#include "UsbInc.h"

#define EDS_COUNT 16
#define TDS_COUNT  0
#define ITDS_COUNT 0
#define UTDS_COUNT 32
#define BPS_COUNT 9

#define HCCA_SIZE 0x100
#define ED_SIZE 0x10
#define TD_SIZE 0x10
#define ITD_SIZE 0x20
#define UTD_SIZE (32+16)
#define BP_SIZE  (128*8)

#define TOTAL_SIZE (HCCA_SIZE + (EDS_COUNT*ED_SIZE) + (TDS_COUNT*TD_SIZE) + (ITDS_COUNT*ITD_SIZE))

static volatile __align(256) byte usb_buf[TOTAL_SIZE] __attribute((section("AHBSRAM0"),aligned));  //256 bytes aligned!
static volatile __align(32) uint8_t usb_utdBuf[UTDS_COUNT*UTD_SIZE] __attribute((section("AHBSRAM0"),aligned));

static volatile byte* usb_hcca;  //256 bytes aligned!

static volatile byte* usb_edBuf;  //4 bytes aligned!

static byte usb_edBufAlloc[EDS_COUNT] __attribute((section("AHBSRAM0"),aligned));
static uint8_t usb_utdBufAlloc[UTDS_COUNT] __attribute((section("AHBSRAM0"),aligned));
static uint8_t usb_bpBufAlloc[BPS_COUNT] __attribute((section("AHBSRAM0"),aligned));
static uint8_t usb_bpBuf[BP_SIZE*BPS_COUNT] __attribute((section("AHBSRAM0"),aligned));

static void utd_init()
{
    DBG_ASSERT(sizeof(HCTD) == 16);
    DBG_ASSERT(sizeof(HCITD) == 32);
    DBG_ASSERT(sizeof(HCUTD) == 48);

    DBG_ASSERT(((uint32_t)usb_utdBuf % 16) == 0);
    DBG_ASSERT(((uint32_t)usb_utdBuf % 32) == 0);

    DBG_ASSERT((uint32_t)usb_utdBufAlloc >= 0x2007c000);
    DBG_ASSERT((uint32_t)&usb_utdBufAlloc[UTDS_COUNT] <= 0x2007ffff);

    DBG_ASSERT((uint32_t)usb_utdBuf >= 0x2007c000);
    DBG_ASSERT((uint32_t)&usb_utdBuf[UTD_SIZE*UTDS_COUNT] <= 0x2007cfff);

    memset(usb_utdBufAlloc, 0x00, UTDS_COUNT);
}

static void pb_init()
{
    memset(usb_bpBufAlloc, 0x00, BPS_COUNT);

    DBG_ASSERT((uint32_t)usb_bpBufAlloc >= 0x2007c000);
    DBG_ASSERT((uint32_t)&usb_bpBufAlloc[BPS_COUNT] <= 0x2007ffff);
    DBG_ASSERT((uint32_t)usb_bpBuf >= 0x2007c000);
    DBG_ASSERT((uint32_t)(&usb_bpBuf[BP_SIZE*BPS_COUNT]) <= 0x2007ffff);
}

void usb_mem_init()
{
  usb_hcca = usb_buf;
  usb_edBuf = usb_buf + HCCA_SIZE;
  memset(usb_edBufAlloc, 0, EDS_COUNT);

  utd_init();
  pb_init();

  DBG("--- Memory Map --- \n");
  DBG("usb_hcca       =%p\n", usb_hcca);
  DBG("usb_edBuf      =%p\n", usb_edBuf);
  DBG("usb_utdBuf     =%p\n", usb_utdBuf);
  DBG("usb_edBufAlloc =%p\n", usb_edBufAlloc);
  DBG("usb_utdBufAlloc=%p\n", usb_utdBufAlloc);
  DBG("usb_bpBuf      =%p\n", usb_bpBuf);
  DBG("               =%p\n", &usb_bpBuf[BP_SIZE*BPS_COUNT]);
  DBG_ASSERT(((uint32_t)usb_hcca % 256) == 0);
  DBG_ASSERT(((uint32_t)usb_edBuf % 16) == 0);
  DBG_ASSERT(((uint32_t)usb_utdBuf % 32) == 0);
}

volatile byte* usb_get_hcca()
{
  return usb_hcca;
}

volatile byte* usb_get_ed()
{
  int i;
  for(i = 0; i < EDS_COUNT; i++)
  {
    if( !usb_edBufAlloc[i] )
    {
      usb_edBufAlloc[i] = 1;
      return usb_edBuf + i*ED_SIZE;
    }
  }
  return NULL; //Could not alloc ED
}

static uint8_t* usb_get_utd(int al)
{
    DBG_ASSERT(al == 16 || al == 32); // GTD or ITD
    if (al == 16) {
        for(int i = 1; i < UTDS_COUNT; i += 2) {
            if (usb_utdBufAlloc[i] == 0) {
                uint32_t p = (uint32_t)usb_utdBuf + i * UTD_SIZE;
                if ((p % al) == 0) {
                    usb_utdBufAlloc[i] = 1;
                    DBG_ASSERT((p % al) == 0);
                    return (uint8_t*)p;
                }
            }
        }
    }
    for(int i = 0; i < UTDS_COUNT; i++) {
        if (usb_utdBufAlloc[i] == 0) {
            uint32_t p = (uint32_t)usb_utdBuf + i * UTD_SIZE;
            if ((p % al) == 0) {
                usb_utdBufAlloc[i] = 1;
                DBG_ASSERT((p % al) == 0);
                return (uint8_t*)p;
            }
        }
    }
    return NULL;
}

volatile byte* usb_get_td(uint32_t endpoint)
{
    DBG_ASSERT(endpoint);
    uint8_t* td = usb_get_utd(16);
    if (td) {
        HCUTD* utd = (HCUTD*)td;
        memset(utd, 0x00, sizeof(HCTD));
        utd->type = 1;
        utd->UsbEndpoint = endpoint;
    }
    return td;
}

volatile byte* usb_get_itd(uint32_t endpoint)
{
    DBG_ASSERT(endpoint);
    uint8_t* itd = usb_get_utd(32);
    if (itd) {
        HCUTD* utd = (HCUTD*)itd;
        memset(utd, 0x00, sizeof(HCITD));
        utd->type = 2;
        utd->UsbEndpoint = endpoint;
    }
    return itd;
}

volatile byte* usb_get_bp(int size)
{
  DBG_ASSERT(size >= 128 && size <= BP_SIZE);
  if (size > BP_SIZE)
      return NULL;
      
  for(int i = 0; i < BPS_COUNT; i++)
  {
    if( !usb_bpBufAlloc[i] )
    {
      usb_bpBufAlloc[i] = 1;
      return usb_bpBuf + i*BP_SIZE;
    }
  }
  return NULL; //Could not alloc Buffer Page
}

int usb_bp_size()
{
    return BP_SIZE; 
}

void usb_free_ed(volatile byte* ed)
{
  int i;
  i = (ed - usb_edBuf) / ED_SIZE;
  usb_edBufAlloc[i] = 0;
}

static void usb_free_utd(volatile uint8_t* utd)
{
  DBG_ASSERT(utd >= usb_utdBuf);
  DBG_ASSERT(utd <= (usb_utdBuf+UTD_SIZE*(UTDS_COUNT-1)));
  DBG_ASSERT(((uint32_t)utd % 16) == 0);
  int i = (utd - usb_utdBuf) / UTD_SIZE;
  DBG_ASSERT(usb_utdBufAlloc[i] == 1);
  usb_utdBufAlloc[i] = 0;
}

void usb_free_td(volatile byte* td)
{
    usb_free_utd(td);
    return;
}

void usb_free_itd(volatile byte* itd)
{
    usb_free_utd(itd);
    return;
}

void usb_free_bp(volatile byte* bp)
{
  DBG_ASSERT(bp >= usb_bpBuf);
  int i;
  i = (bp - usb_bpBuf) / BP_SIZE;
  DBG_ASSERT(usb_bpBufAlloc[i] == 1);
  usb_bpBufAlloc[i] = 0;
}

bool usb_is_td(volatile byte* td)
{
    DBG_ASSERT(td);
    HCUTD* utd = (HCUTD*)td;
    DBG_ASSERT(utd->type != 0);
    return utd->type == 1;
}

bool usb_is_itd(volatile byte* itd)
{
    DBG_ASSERT(itd);
    HCUTD* utd = (HCUTD*)itd;
    DBG_ASSERT(utd->type != 0);
    return utd->type == 2;
}
