#include "mbed.h"
#include "FreescaleIAP.h"

//Could be nicer, but for now just erase all preceding sectors
#define NUM_SECTORS        28
#define TIMEOUT            10000000
#define BUFFER_SIZE        16

void setupserial();
void write(char *value);
/*KL05 32kB Memmory Map
0x0000 : 0x7FFF
32 1kB  Flash Secotrs
[1:28]  Program Memory
[29]    Bootloader
[30]    IAP
[31]    IAP
[32]    Flash Storage

0x7000 Sector 29 Start Addr
0x7400 Sector 30 Start Addr
0x7800 Sector 31 Start Addr
0x7C00 Sector 32 Start Addr

0x7000 bootloader
0x7080 setupserial
0x70A0 write
0x7260 erase_sector
0x7300 program_flash
0x7500 flash_size
0x7600 program_word
0x7700 run_command
0x7800 check_boundary
0x7900 check_align
0x7A00 verify_erased
0x7B00 check_error
*/
//KL05 Version
__attribute__((section(".ARM.__at_0x7000"))) void bootloader(void)
{
    setupserial();
    write("\n\n\rBootloader\r\n");
    write("Continue? (y/n)");
    
    //Wait until data arrived, if it is 'y', continue
    while(!(UART0->S1 & UART0_S1_RDRF_MASK));
    if (UART0->D != 'y')
        return;
    
    //Erase all sectors we use for the user program
    write("Erasing sectors!\r\n");
    for (int i = 0; i<NUM_SECTORS; i++)
        erase_sector(SECTOR_SIZE * i);

    write("Done erasing, send file!\r\n");

    char buffer[BUFFER_SIZE];
    uint32_t count = 0;
    uint8_t buffercount = 0;
    uint32_t timeout = 0;
    
    //Wait until data is sent
    while(!(UART0->S1 & UART0_S1_RDRF_MASK));
    
    //Data receive loop
    while(1) {
        //Check if there is new data
        if (UART0->S1 & UART0_S1_RDRF_MASK) {
            //Place data in buffer
            buffer[buffercount] = UART0->D;
            buffercount++;
            
            //Reset timeout
            timeout = 0;

            //We write per BUFFER_SIZE chars
            if (buffercount == BUFFER_SIZE) {
                //NMI Handler is at bytes 8-9-10-11, we overwrite this to point to bootloader function
                if (count == 0) {
                    buffer[8] = 0x01;
                    buffer[9] = 0x00;
                    buffer[10] = 0x01;
                    buffer[11] = 0x00;
                }
                
                //Program the buffer into the flash memory
                if (program_flash(count, buffer, BUFFER_SIZE) != 0) {
                    write("Error!\r\n");   
                    break;
                }
                
                //Reset buffercount for next buffer
                write("#");
                buffercount = 0;
                count += BUFFER_SIZE;
            }
        } else {
            //No new data, increase timeout
            timeout++;            
            //We have received no new data for a while, assume we are done
            if (timeout > TIMEOUT) {
                //If there is data left in the buffer, program it
                if (buffercount != 0) {
                    for (int i = buffercount; i<BUFFER_SIZE; i++) {
                        buffer[i] = 0xFF;
                    }
                    program_flash(count, buffer, BUFFER_SIZE);
                }
                break;          //We should be done programming :D
            }
        }
    }
    write("Done programming!\r\n");
    NVIC_SystemReset();
    
    //Shouldn't arrive here
    while(1);
}

__attribute__((section(".ARM.__at_0x7080"))) static void setupserial(void) {
        //Setup USBRX/USBTX pins (PTB1/PTB2)
        //Disable UART0
//        UART0->C2 &= ~(UARTLP_C2_RE_MASK | UARTLP_C2_TE_MASK);
        //Enable Port B Clock
        SIM->SCGC5 |= 1 <<SIM_SCGC5_PORTB_SHIFT;                   
        //Select MCGFLLCLK clock
        SIM->SOPT2 |= 1 <<SIM_SOPT2_UART0SRC_SHIFT;
        //Select Pins PB1 & PB2 to their ALT3 function (RX & TX respectively)
        PORTB->PCR[1] = (PORTB->PCR[1] & ~0x700) | (3 << 8);
        PORTB->PCR[2] = (PORTB->PCR[2] & ~0x700) | (3 << 8);
        //Set UART0 Clock to be enabled
        SIM->SCGC4 |= SIM_SCGC4_UART0_MASK;
//      KL05's pclk def
//      uint32_t PCLK = SystemCoreClock * (1u + ((SIM->CLKDIV1 & SIM_CLKDIV1_OUTDIV1_MASK) >> SIM_CLKDIV1_OUTDIV1_SHIFT));
        //Set UART Baud Rate Register
//        uint32_t PCLK = SystemCoreClock;
//        uint16_t DL = PCLK / (16 * 9600); //Want 9600 Baudrate
//        UART0->BDH = ((DL >> 8) & 0x1f);
//        UART0->BDL = ((DL >> 0) & 0xff);
        //Value's gathered expirimentally   
        UART0->BDH = 1;
        UART0->BDL = 56;   
        //Enable UART0
        UART0->C2 |= (UARTLP_C2_RE_MASK | UARTLP_C2_TE_MASK);
}

__attribute__((section(".ARM.__at_0x70A0"))) static void write(char *value)
{
        int i = 0;
        //Loop through string and send everything
        while(*(value+i) != '\0') {
            while(!(UART0->S1 & UART0_S1_TDRE_MASK));
            UART0->D = *(value+i); 
            i++;
        }
    }
