/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include <stdio.h>
#include <errno.h>

// Block devices
#include "QSPIExtFlashBlockDevice.h"

// File systems
#include "LittleFileSystem.h"
//#include "FATFileSystem.h"


// Physical block device, can be any device that supports the BlockDevice API
QSPIExtFlashBlockDevice m_cBlockDevice;
// File system declaration
LittleFileSystem m_cFileSystem("fs");
//FATFileSystem m_cFileSystem("fs", &m_cBlockDevice);



// Entry point for the example
int main()
{
    int nRc = 0;
    int err;
    bool bIsToEraseChip = false;
    bool bIsToFirstFormat = false;

    printf("===> Qspi External flash filesystem example ---\n");
    if(bIsToEraseChip)
    {
        nRc = m_cBlockDevice.init();
        if(nRc != 0)
        {
            error("BlockDevice init error: %d\n", nRc);
        }
        nRc = m_cBlockDevice.EraseAllFlashMemory();
        if(nRc != 0)
        {
            error("BlockDevice init error: %d\n", nRc);
        }
        nRc = m_cBlockDevice.deinit();
    }

    // Try to mount the filesystem
    if(!bIsToFirstFormat)
    {
        printf("Mounting the filesystem... \n");
        fflush(stdout);
        err = m_cFileSystem.mount((BlockDevice *) &m_cBlockDevice);
        printf("Mount %s, err=%d\n", (err ? "Fail :(" : "OK"), err);
        if(err)
        {
          printf("No filesystem found, ");
        }
    }
    else
    {
        err = -1;
    }
    if (err)
    {
        // Reformat if we can't mount the filesystem
        // this should only happen on the first boot
        printf("Formatting... \n");
        fflush(stdout);
        err = m_cFileSystem.reformat((BlockDevice *) &m_cBlockDevice);
        printf("Format %s, err=%d\n", (err ? "Fail :(" : "OK"), err);
        printf("Formatting terminated... \n");
    }

    // Open the numbers file
    if(err == 0)
    {
        printf("Opening \"/fs/numbers.txt\"... ");
        fflush(stdout);
        FILE *f = fopen("/fs/numbers.txt", "r+");
        printf("%s\n", (!f ? "Fail :(" : "OK"));
        if (!f) {
            // Create the numbers file if it doesn't exist
            printf("No file found, creating a new file... ");
            fflush(stdout);
            f = fopen("/fs/numbers.txt", "w+");
            printf("%s\n", (!f ? "Fail :(" : "OK"));
            if (!f) {
                error("error: %s (%d)\n", strerror(errno), -errno);
            }
    
            for (int i = 0; i < 10; i++) {
                printf("\rWriting numbers (%d/%d)... ", i, 10);
                fflush(stdout);
                err = fprintf(f, "    %d\n", i);
                if (err < 0) {
                    printf("Fail :(\n");
                    error("error: %s (%d)\n", strerror(errno), -errno);
                }
            }
            printf("\rWriting numbers (%d/%d)... OK\n", 10, 10);
    
            printf("Seeking file... ");
            fflush(stdout);
            err = fseek(f, 0, SEEK_SET);
            printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
            if (err < 0) {
                error("error: %s (%d)\n", strerror(errno), -errno);
            }
        }
    
        // Go through and increment the numbers
        for (int i = 0; i < 10; i++) {
            printf("\rIncrementing numbers (%d/%d)... ", i, 10);
            fflush(stdout);
    
            // Get current stream position
            long pos = ftell(f);
    
            // Parse out the number and increment
            int32_t number;
            fscanf(f, "%d", &number);
            number += 1;
    
            // Seek to beginning of number
            fseek(f, pos, SEEK_SET);
        
            // Store number
            fprintf(f, "    %d\n", number);
        }
        printf("\rIncrementing numbers (%d/%d)... OK\n", 10, 10);
    
        // Close the file which also flushes any cached writes
        printf("Closing \"/fs/numbers.txt\"... ");
        fflush(stdout);
        err = fclose(f);
        printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
        if (err < 0) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
        
        // Display the root directory
        printf("Opening the root directory... ");
        fflush(stdout);
        DIR *d = opendir("/fs/");
        printf("%s\n", (!d ? "Fail :(" : "OK"));
        if (!d) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    
        printf("root directory:\n");
        while (true) {
            struct dirent *e = readdir(d);
            if (!e) {
                break;
            }
    
            printf("    %s\n", e->d_name);
        }
    
        printf("Closing the root directory... ");
        fflush(stdout);
        err = closedir(d);
        printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
        if (err < 0) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    
        // Display the numbers file
        printf("Opening \"/fs/numbers.txt\"... ");
        fflush(stdout);
        f = fopen("/fs/numbers.txt", "r");
        printf("%s\n", (!f ? "Fail :(" : "OK"));
        if (!f) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    
        printf("numbers:\n");
        while (!feof(f)) {
            int c = fgetc(f);
            printf("%c", c);
        }
    
        printf("\rClosing \"/fs/numbers.txt\"... ");
        fflush(stdout);
        err = fclose(f);
        printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
        if (err < 0)
        {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    
        // Tidy up
        printf("Unmounting... ");
        fflush(stdout);
        err = m_cFileSystem.unmount();
        printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
        if (err < 0)
        {
            error("error: %s (%d)\n", strerror(-err), err);
        }
    }
    m_cBlockDevice.deinit();
        
    printf("==> Qspi External flash filesystem example done!\n");
}

