/* QSPIExtFlashBlockDevice.h
 * QSPI external flash block device for file system on external flash
 * for DISCO_L475VG_IOT01
 * 
 * Doron Raifman, draifman@gmail.com
 */
#ifndef QSPIExtFlashBlockDevice_H
#define QSPIExtFlashBlockDevice_H

#include <mbed.h>
#include <BlockDevice.h>

#include "stm32l475e_iot01_qspi.h"


#define QSPIExtFlashBlockDeviceTimeout          1000    // Timeout [mSec]
#define QSPIExtFlashBlockDeviceEraseTimeout     10000   // Timeout [0.1 mSec]
#define QSPIExtFlashBlockDevice_VERBOSE         0       // Verbose Debug prints
#define QSPIExtFlashBlockDevice_DEBUG           0       // Debug prints
#define QSPIExtFlashBlockDevice_ERRORS          1       // Errors prints

// Error codes
#define RC_QSPIExtFlash_OK                      0
#define RC_QSPIExtFlash_ErrTimeout              -1
#define RC_QSPIExtFlash_ErrBadDevice            -2
#define RC_QSPIExtFlash_ErrMemAloc              -3
#define RC_QSPIExtFlash_ErrIO                   -4
#define RC_QSPIExtFlash_ErrInit                 -5

 
/*
 * BlockDevice for QSPI external flash devices on ST DISCO_L476VG
 */
class QSPIExtFlashBlockDevice : public BlockDevice
{
public:
    /** Creates a BlockDevice on a QSPI bus
     */
    QSPIExtFlashBlockDevice();
    virtual ~QSPIExtFlashBlockDevice();

    /** Initialize a block device
     *
     *  @return         0 on success or a negative error code on failure
     */
    virtual int init();

    /** Deinitialize a block device
     *
     *  @return         0 on success or a negative error code on failure
     */
    virtual int deinit();

    /** Read blocks from a block device
     *
     *  @param buffer   Buffer to write blocks to
     *  @param addr     Address of block to begin reading from
     *  @param size     Size to read in bytes, must be a multiple of read block size
     *  @return         0 on success, negative error code on failure
     */
    virtual int read(void *buffer, bd_addr_t addr, bd_size_t size);

    /** Program blocks to a block device
     *
     *  The blocks must have been erased prior to being programmed
     *
     *  @param buffer   Buffer of data to write to blocks
     *  @param addr     Address of block to begin writing to
     *  @param size     Size to write in bytes, must be a multiple of program block size
     *  @return         0 on success, negative error code on failure
     */
    virtual int program(const void *buffer, bd_addr_t addr, bd_size_t size);

    /** Erase blocks on a block device
     *
     *  The state of an erased block is undefined until it has been programmed
     *
     *  @param addr     Address of block to begin erasing
     *  @param size     Size to erase in bytes, must be a multiple of erase block size
     *  @return         0 on success, negative error code on failure
     */
    virtual int erase(bd_addr_t addr, bd_size_t size);

    /** Erase all flash memory
     *
     *  @return         0 on success, negative error code on failure
     */
    int EraseAllFlashMemory();

    /** Get the size of a readable block
     *
     *  @return         Size of a readable block in bytes
     */
    virtual bd_size_t get_read_size() const;

    /** Get the size of a programable block
     *
     *  @return         Size of a programable block in bytes
     *  @note Must be a multiple of the read size
     */
    virtual bd_size_t get_program_size() const;

    /** Get the size of a eraseable block
     *
     *  @return         Size of a eraseable block in bytes
     *  @note Must be a multiple of the program size
     */
    virtual bd_size_t get_erase_size() const;

    /** Get the total size of the underlying device
     *
     *  @return         Size of the underlying device in bytes
     */
    virtual bd_size_t size() const;
private:
    int _SyncLock() const;
    int _SyncUnlock() const;
    int _WaitForWrite(const char *ptCaller);
    
private:
    // Device configuration discovered through init
    QSPI_Info m_sQSPIExtFlash_Info;
};


#endif  /* QSPIExtFlashBlockDevice_H */
