#define logMessage printf
#define MQTTCLIENT_QOS2 1

#include "mbed.h"
#include "easy-connect.h"
#include "MQTTNetwork.h"
#include "MQTTmbed.h"
#include "MQTTClient.h"

#define ADT7420_TEMP_REG (0x00) // Temperature Register
#define ADT7420_CONF_REG (0x03) // Configuration Register
#define EVAL_ADT7420_ADDR (0x48) // EVAL_ADT7420 i2c address


/* connect this pin to both the CH_PD (aka EN) & RST pins on the ESP8266 just in case */
#define WIFI_HW_RESET_PIN D4

/* See if you can try using a hostname here */
#define MQTT_BROKER_ADDR "192.168.0.14"
#define MQTT_BROKER_PORT 1883

DigitalOut wifiHwResetPin(WIFI_HW_RESET_PIN);
DigitalOut led2(LED2);
Serial pc(USBTX, USBRX); // computer to mbed boardSerial esp(D1, D0); 
I2C i2c(I2C_SDA, I2C_SCL);

volatile bool flag = false;
Mutex mqttMtx;

char* topic1 = "LED";
char* topic2 = "TEMP"; 

void messageArrived(MQTT::MessageData& md)
{
    MQTT::Message &message = md.message;

    /* our messaging standard says the first byte denotes which thread to fwd to */
    logMessage("Message arrived: qos %d, retained %d, dup %d, packetid %d\r\n",
    message.qos, message.retained, message.dup, message.id);
    logMessage("Payload %.*s\r\n", message.payloadlen, (char*)message.payload);

    char fwdTarget = ((char *)message.payload)[0];
    switch(fwdTarget)
    {
        case '0': // turn off
        led2 = 0;
        break;
        
        case '1': //turn on
        led2 = 1;
        break;
    
        default:
        pc.printf("Unknown MQTT message\n");
        break;
    }
}

int main(int argc, char* argv[])
{
    float version = 0.5;
    pc.baud(115200);
    logMessage("MQTT example: version is %.2f\r\n", version);
    
    wait(0.2); //delay startup
    pc.printf("Resetting ESP8266 Hardware...\r\n");

    wifiHwResetPin = 0;
    wait_ms(500);
    wifiHwResetPin = 1;
    pc.printf("Starting MQTT example with an ESP8266 wifi device using Mbed OS.\r\n");
    pc.printf("Attempting to connect to access point...\r\n");

    NetworkInterface* network = easy_connect(true);
    if (!network) {
        pc.printf("Error in east connection\r\n");
        return -1;
    }
    
    MQTTNetwork mqttNetwork(network);
    MQTT::Client<MQTTNetwork, Countdown> client(mqttNetwork);
    logMessage("Connecting to %s:%d\r\n", MQTT_BROKER_ADDR, MQTT_BROKER_PORT);
    int rc = mqttNetwork.connect(MQTT_BROKER_ADDR, MQTT_BROKER_PORT);

    if (rc != 0)
        logMessage("rc from TCP connect is %d\r\n", rc);

    MQTTPacket_connectData data = MQTTPacket_connectData_initializer;
    data.MQTTVersion = 3;
    data.clientID.cstring = "mbed-kdh";
    data.username.cstring = "kdh";
    data.password.cstring = "2ehd2gh4";

    if ((rc = client.connect(data)) != 0)
        logMessage("rc from MQTT connect is %d\r\n", rc);

    if ((rc = client.subscribe(topic1, MQTT::QOS0, messageArrived)) != 0)
        logMessage("rc from MQTT subscribe is %d\r\n", rc);

    int count = 0;

    int status;
    float temp;
    char data_write[2];
    char data_read[2];
    
    i2c.frequency(100000); // 100 k

    // 13-bit resolution and 1 SPS mode
    data_write[0] = ADT7420_CONF_REG;
    data_write[1] = 0x40;
    status = i2c.write((EVAL_ADT7420_ADDR << 1), data_write, 2);

    if (status != 0) { // error
        char buf[100];
        MQTT::Message message;
        
        pc.printf("I2C configuration error!!\r\n");
        sprintf(buf, "{ \"actions\": [ { \"name\": 'error' , \"parameter\": \"NaN\" } ] }");
        pc.printf("%s\r\n", buf);         
                
        message.qos = MQTT::QOS2;
        message.retained = false;
        message.dup = false;
        message.payload = (void*)buf;
        message.payloadlen = strlen(buf)+1;
        mqttMtx.lock();
                
        rc = client.publish(topic2, message);
        mqttMtx.unlock();
        
        while(1) {
        }
    }

    while(1) {
        Thread::wait(1000);
        char buf[100];
        MQTT::Message message;

        if (count == 5) {
            
            // Read temperature register
            data_write[0] = ADT7420_TEMP_REG;
            i2c.write((EVAL_ADT7420_ADDR << 1), data_write, 1, 0);
            i2c.read(((EVAL_ADT7420_ADDR << 1) | 0x01), data_read, 2, 0);
    
            // Calculate temperature value in Celcius
            int tempval = ((int)data_read[0] << 8) | data_read[1];
            tempval >>= 3;

            // tempval = 0x1e70; //test value for a negative temperature (-25.0)
            if ((tempval & 0x1000) > 0) { //negative value
                temp = (tempval - 8192)/16.0;
            } else {
                temp = tempval/16.0;
            }
            
            pc.printf("Publish: TEMP\r\n");
            sprintf(buf, "%4.2f", temp);
            pc.printf("%s\r\n", buf);  
                
            message.qos = MQTT::QOS2;
            message.retained = false;
            message.dup = false;
            message.payload = (void*)buf;
            message.payloadlen = strlen(buf)+1;
            mqttMtx.lock();
                
            rc = client.publish(topic2, message);
            mqttMtx.unlock();
            count = 0;
        }

        if(!client.isConnected())
            NVIC_SystemReset(); // soft reset

        /* yield() needs to be called at least once per keepAliveInterval */
        client.yield(10);
        count++;
    }
    return 0;
}