/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef MMA8491Q_H
#define MMA8491Q_H

#include "mbed.h"

/**
* MMA8491Q accelerometer example
*
* @code
* #include "mbed.h"
* #include "MMA8491Q.h"
* 
* #define MMA8451_I2C_ADDRESS (0x55<<1)
*
* Serial pc(USBTX, USBRX);
* 
* int main(void) {
* 
*    MMA8491Q acc(PTE0, PTE1, MMA8451_I2C_ADDRESS, PTA13);
*    
*    PwmOut rled(LED_RED);
*    PwmOut gled(LED_GREEN);
*    PwmOut bled(LED_BLUE);
*    
*    float accData[3];
*
*    while (true) {       
*        acc.getAccAllAxis(accData);
*        accX = accData[0];
*        accY = accData[1];
*        accZ = accData[2];
*         
*        pc.printf("X = %5.3f", accX);
*        pc.printf("  Y = %5.3f", accY);
*        pc.printf("  Z = %5.3f\r\n", accZ);
* 
*        wait(1.0);
*   }
*}
*
* @endcode
*/

class MMA8491Q
{
public:
  /**
  * MMA8491Q constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  * @param en EN pin
  * @param Xout digital output pin
  * @param Yout digital output pin
  * @param Zout digital output pin
  */
  MMA8491Q(PinName sda, PinName scl, int addr, PinName en);
  
  MMA8491Q(PinName sda, PinName scl, int addr, PinName en, PinName Xout, PinName Yout, PinName Zout);

  /**
  * MMA8491Q destructor
  */
  ~MMA8491Q();

  /**
   * Get X axis acceleration
   *
   * @returns X axis acceleration
   */
  float getAccX();

  /**
   * Get Y axis acceleration
   *
   * @returns Y axis acceleration
   */
  float getAccY();

  /**
   * Get Z axis acceleration
   *
   * @returns Z axis acceleration
   */
  float getAccZ();

  /**
   * Get XYZ axis acceleration
   *
   * @param res array where acceleration data will be stored
   * @returns true if sensor was ready and data was read
   */
  uint8_t getAccAllAxis(float * res);
  

private:
  I2C m_i2c;
  int m_addr;          //MMA8491Q address
  DigitalOut m_en;     //MMA8491Q ENABLE pin
  PinName m_xout;
  PinName m_yout;
  PinName m_zout; 
  uint8_t isDataAvailable( uint8_t mask);
  float intAccToFloat(uint8_t * data);
  void toggleEN(void);
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);
  int16_t getAccAxis(uint8_t addr);
  

};

#endif
