#include "mbed.h"

//---------------------------------------------------------
//Enter amount of minutes between each log
int minutesPerLog = 15;

//Enter amount of seconds for the log screen to show
int logTimer = 20;
//---------------------------------------------------------

//I2C pins
I2C i2c(PB_9, PB_8); //SDA, SCL | D14 , D15

int getTemp(void);
void displayLog(int *);
void initScreen(void);

LCD_DISCO_F746NG LCDscreen;
TS_StateTypeDef touchStatus;
TS_DISCO_F746NG LCDtouchScreen;

/**
  * @brief  Displays the temperature
  * @retval None
  */
void tempDisplay(void)
{
    //If the log is recording every 0 minutes do not show log
    if (minutesPerLog == 0) {
        return;
    }

    initScreen();
    i2c.frequency(100000);

    int counter = 0;
    unsigned short StringX = 0;
    unsigned short StringY = 250;
    unsigned char helpDisplay[100] = {"Hold screen for Log"};
    unsigned char tempDisplay[100];
    int tempLog[10] = {0};

    //Reset screen after log display
back:
    LCDscreen.Clear(LCD_COLOR_BLUE);
    LCDscreen.DisplayStringAt(0, 0, helpDisplay, LEFT_MODE);

    //Main loop
    while(1) {
        counter++;
        //Grabs temp and puts it into display portion of log [0], also adds one
        //second delay
        int temp = getTemp();
        tempLog[0] = temp;

        //Checks to see if time has been met for a log update
        if (counter >= (minutesPerLog * 60)) {
            for (int i = 9; i > 0; i--) {
                tempLog[i] = tempLog[i - 1];
            }
            counter = 0;
        }

        //Displays the current temperature
        if (temp < 100) {
            sprintf((char*)tempDisplay, "Current Temperature = %dF", temp);
        } else if (temp >= 100) {
            sprintf((char*)tempDisplay, "Current Temperature =%dF", temp);
        }
        LCDscreen.DisplayStringAt(StringX, StringY, tempDisplay, LEFT_MODE);

        //Checks for touch to display log
        LCDtouchScreen.GetState(&touchStatus);
        if (touchStatus.touchDetected) {
            displayLog(tempLog);
            counter = counter + logTimer;
            goto back;
        }
    }
}

/**
  * @brief  Retrieves temperature from chip
  * @retval Temperature
  */
int getTemp(void)
{
    char tempData[2];
    char getTemp[1] = {0xAA};
    char stopConvert[1] = {0x22};
    char startConvert[1] = {0xEE};
    int temp;

    //Grabs the temperature from the DS1621, converts to F and returns value
    //Convert tells the DS1621 to update its temperature reading
    //Read and Write have different addresses because I2C communication address
    //uses a 1 for write and a 0 for read at end of binary address
    i2c.write(145, startConvert, 1, false);
    wait(1);
    i2c.write(145, stopConvert, 1, false);
    i2c.write(145, getTemp, 1, false);
    i2c.read(144, tempData, 1);
    temp = (tempData[0] * 1.8) + 32;
    return (temp);
}

/**
  * @brief  Displays the temperature log
  * @param  tempLog[10]: Array of temperature logs
  * @retval None
  */
void displayLog(int tempLog[10])
{
    LCDscreen.SetBackColor(LCD_COLOR_BLUE);
    LCDscreen.SetTextColor(LCD_COLOR_WHITE);

    unsigned char logDisplay[500];
    unsigned char helpDisplay2[100];

    sprintf((char*)helpDisplay2, "Log is updated every %d minutes", minutesPerLog);

    //Displays 10 logs, the first log is the current temperature. If the time
    //of one log is divisible by 60 it will note that log as being one hour ago
    //Each log is added to the master string "logDisplay"
    LCDscreen.Clear(LCD_COLOR_BLUE);
    sprintf((char*)logDisplay, "Log:");
    LCDscreen.DisplayStringAt(0, 0, logDisplay, LEFT_MODE);
    for (int i = 0; i < 10; i++) {
        if (i == 0) {
            sprintf((char*)logDisplay, "%dF - Current Temperature", tempLog[i]);
        } else if (i == (60 / minutesPerLog)) {
            sprintf((char*)logDisplay, "%d: %dF - One Hour Ago", i, tempLog[i]);
        } else {
            sprintf((char*)logDisplay, "%d: %dF", i, tempLog[i]);
        }
        LCDscreen.DisplayStringAt(0, (i + 1) * 20, logDisplay, LEFT_MODE);
    }
    LCDscreen.SetFont(&Font20);
    LCDscreen.DisplayStringAt(0, 240, helpDisplay2, LEFT_MODE);
    LCDscreen.SetFont(&Font24);

    wait(logTimer);
}

/**
  * @brief  Initializes display
  * @retval None
  */
void initScreen(void)
{
    LCDscreen.Clear(LCD_COLOR_WHITE);
    LCDscreen.SetBackColor(LCD_COLOR_BLUE);
    LCDscreen.SetTextColor(LCD_COLOR_WHITE);
    LCDscreen.SetFont(&Font24);
    LCDscreen.Clear(LCD_COLOR_BLUE);
}