#ifndef MAX30101_H
#define MAX30101_H
#include "mbed.h"

#define MAX30101_I2C_ADDRESS (0xAE<<1) //pins SA0,SA1=0

#define MAXIM_BYTES_PER_ADC_VALUE ( 3 )
#define MAXIM_FIFO_DEPTH          ( 32 )

#define MAX30101_InterruptStatus1                       0x00
#define MAX30101_InterruptStatus2                       0x01
#define MAX30101_InterruptEnable1                       0x02
#define MAX30101_InterruptEnable2                       0x03

#define MAX30101_FIFO_WR_PTR                            0x04
#define MAX30101_FIFO_OV_PTR                            0x05
#define MAX30101_FIFO_RD_PTR                            0x06
#define MAX30101_FIFO_DATA                              0x07
#define MAX30101_FIFOConfiguration                      0x08
#define MAX30101_ModeConfiguration                      0x09
#define MAX30101_SPO2Configuration                      0x0A
#define MAX30101_LED1RedPulseAmplitude                  0x0C
#define MAX30101_LED2IrPulseAmplitude                   0x0D
#define MAX30101_LED3GreenPulseAmplitude                0x0E
#define MAX30101_ProximityModePulseAmplitude            0x10
#define MAX30101_MultiLEDModeControlTime2Time1          0x11
#define MAX30101_MultiLEDModeControlTime4Time3          0x12
#define MAX30101_DieTemperatureInteger                  0x1F
#define MAX30101_DieTemperatureFraction                 0x20
#define MAX30101_DieTemperatureConfig                   0x21
#define MAX30101_ProximityInterruptThreshold            0x30
#define MAX30101_RevisionID                             0xFE
#define MAX30101_DeviceID                               0xFF

/******************************************************************************/
#define A_FULL          0x80
#define PPG_RD          0x40
#define ALC_OVF         0x20
#define PROX_IN         0x10
#define PWR_RDY         0x01

#define DIE_TEMP_RDY    0x02

#define A_FULL_EN       0x80
#define PPG_RDY_EN      0x40
#define ALC_OVF_EN1     0x20
#define ALC_OVF_EN0     0x10

#define DIE_TEMP_RDY_EN 0x02

#define SHDN            0x80
#define RESET           0x40
#define MODE1           0x02
#define MODE0           0x01

#define SPO2_ADC_RGE1   0x40 
#define SPO2_ADC_RGE0   0x20
#define SPO2_SR2        0x10
#define SPO2_SR1        0x08
#define SPO2_SR0        0x04
#define LED_PW1         0x02
#define LED_PW0         0x01

#define TEMP_EN         0x01

/******************************************************************************/

#define MAXIM_SLOT_NUM  2

typedef enum
{
  modeHR        = 0x2,
  modeSPO2      = 0x3,
  modeMultiLED  = 0x7

} max30101_mode_t;

typedef enum
{
  ledPower_69  = 0x0, // ADC -> 15 bit
  ledPower_118 = 0x1, // ADC -> 16 bit
  ledPower_215 = 0x2, // ADC -> 17 bit
  ledPower_411 = 0x3  // ADC -> 18 bit

} max30101_ledpwr_t;


/** HR global status enum */
typedef enum
{
    HEART_RATE_STATUS_SUCCESS,
    HEART_RATE_STATUS_ERROR

} heartRate_status_t;

typedef enum
{
  sampleRate_50   = 0x0,
  sampleRate_100  = 0x1,
  sampleRate_200  = 0x2,
  sampleRate_400  = 0x3,
  sampleRate_800  = 0x4,
  sampleRate_1000 = 0x5,
  sampleRate_1600 = 0x6,
  sampleRate_3200 = 0x7

} max30101_sampleRate_t;

typedef enum
{
  LEDSlotNone  =  0x0,
  LEDSlotRed   =  0x1,
  LEDSlotIR    =  0x2,
  LEDSlotGreen =  0x3

} max30101_LEDSlot_t;

typedef enum
{
  STATUS_MAXIM_SUCCESS,        // success
  STATUS_MAXIM_ERROR,          // fail
  STATUS_MAXIM_PROTOCOL_ERROR, // protocol failure
  STATUS_MAXIM_INIT_ERROR,     // initialization error
  STATUS_MAXIM_PARAM_ERROR,    // invalid parameter is given
  STATUS_MAXIM_OVF_ERROR,      // sensor data overflow
  STATUS_MAXIM_TIMEOUT         // timeout occured

} max30101_status_t;

typedef enum
{
  adcRange2048  = 0x0,
  adcRange4096  = 0x1,
  adcRange8192  = 0x2,
  adcRange16384 = 0x3

} max30101_adcRange_t;

typedef enum
{
  ovsNone = 0x0, // no averaging
  ovs2    = 0x1, // average 2 samples
  ovs4    = 0x2, // average 4 samples
  ovs8    = 0x3, // average 8 samples
  ovs16   = 0x4, // average 16 samples
  ovs32   = 0x5  // average 32 samples

} max30101_ovs_t;

#define HR_SAMPLE_RATE ( 50 ) /** sample rate for MAXIM 30101 sensor */


class MAX30101
{
public:
  /**
  * MAX30101 constructor
  *
  * @param sda SDA pin
  * @param sdl SCL pin
  * @param addr addr of the I2C peripheral
  */
  MAX30101(PinName sda, PinName scl);
 
  /**
  * MAX30101 destructor
  */
  ~MAX30101();

    void enable(void);
     
    void shutdown(void);
    
    max30101_status_t reset(void);
        
    void setMode(max30101_mode_t sensorMode);
    
    void setSPO2_ADC_range(void);
    
    void set_SPO2_sampleRange(void);
    
    void set_LED_PulseWidth(uint8_t pw);
    
    max30101_status_t readRawData(uint8_t* dataBuff, uint8_t* sampleNum );
    float getTemp(void);
    
    float getDieTemp(void);
    
    uint8_t  getRevisionID(void);

    uint8_t getChipID(void);
 
private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);
  
  uint8_t readRegister( uint8_t regaddr);
  void writeRegister(uint8_t regaddr, uint8_t data);
  
  // mode (HR, SpO2, or multi-led)
  max30101_mode_t mode;

  // multiLED slots
  max30101_LEDSlot_t LEDSlot[ MAXIM_SLOT_NUM ];
  uint8_t activeSlots;

  // LED currents
  uint8_t LEDCurrentRed;
  uint8_t LEDCurrentIR;
  uint8_t LEDCurrentGreen;

  // timings
  max30101_sampleRate_t sampleRate;
  max30101_ledpwr_t pulseWidth;
  // ADC range
  max30101_adcRange_t range;
  // averaging
  max30101_ovs_t oversample;
 
};
 
#endif




