/**
 ******************************************************************************
 * @file    nucleo-f401re.c
 * @author  MCD Application Team
 * @version V1.0.0
 * @date    22-April-2014
 * @brief   This file provides set of firmware functions to manage:
 *          - LEDs and push-button available on NUCLEO-F401RE Kit
 *            from STMicroelectronics
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2014 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "nucleo-f401re.h"
//#include "cmsis_os.h"
//#include "stm32f4xx_hal.h"
//#include "stm32f4xx_hal_i2c.h"
//#include "stm32f4xx_hal_i2c_ex.h"


/** @addtogroup BSP
 * @{
 */

/** @addtogroup NUCLEO_F401RE
 * @{
 */

/** @addtogroup NUCLEO_F401RE_LOW_LEVEL 
 * @brief This file provides set of firmware functions to manage Leds and push-button
 *        available on NUCLEO-F401RE Kit from STMicroelectronics.
 * @{
 */

/** @defgroup NUCLEO_F401RE_LOW_LEVEL_Private_TypesDefinitions 
 * @{
 */
/**
 * @}
 */


/** @defgroup NUCLEO_F401RE_LOW_LEVEL_Private_Defines 
 * @{
 */

/**
 * @brief STM32L0XX NUCLEO BSP Driver version number V1.0.0
 */
#define __NUCLEO_F401RE_BSP_VERSION_MAIN   (0x01) /*!< [31:24] main version */
#define __NUCLEO_F401RE_BSP_VERSION_SUB1   (0x00) /*!< [23:16] sub1 version */
#define __NUCLEO_F401RE_BSP_VERSION_SUB2   (0x00) /*!< [15:8]  sub2 version */
#define __NUCLEO_F401RE_BSP_VERSION_RC     (0x00) /*!< [7:0]  release candidate */ 
#define __NUCLEO_F401RE_BSP_VERSION         ((__NUCLEO_F401RE_BSP_VERSION_MAIN << 24)\
        |(__NUCLEO_F401RE_BSP_VERSION_SUB1 << 16)\
        |(__NUCLEO_F401RE_BSP_VERSION_SUB2 << 8 )\
        |(__NUCLEO_F401RE_BSP_VERSION_RC))

/**
 * @brief LINK SD Card
 */
#define SD_DUMMY_BYTE            0xFF    
#define SD_NO_RESPONSE_EXPECTED  0x80

/**
 * @}
 */

/** @defgroup NUCLEO_F401RE_LOW_LEVEL_Private_Macros
 * @{
 */
/**
 * @}
 */

/** @defgroup NUCLEO_F401RE_LOW_LEVEL_Private_Variables
 * @{
 */
GPIO_TypeDef* GPIO_PORT[LEDn] = {LED2_GPIO_PORT};

const uint16_t GPIO_PIN[LEDn] = {LED2_PIN};

GPIO_TypeDef*  BUTTON_PORT[BUTTONn] = {KEY_BUTTON_GPIO_PORT};
const uint16_t BUTTON_PIN[BUTTONn] = {KEY_BUTTON_PIN};
const uint16_t BUTTON_IRQn[BUTTONn] = {KEY_BUTTON_EXTI_IRQn};

/**
 * @brief BUS variables
 */

//uint32_t SpixTimeout = NUCLEO_SPIx_TIMEOUT_MAX; /*<! Value of Timeout when SPI communication fails */
//static SPI_HandleTypeDef hnucleo_Spi;
//static ADC_HandleTypeDef hnucleo_Adc;

/* ADC channel configuration structure declaration */
//static ADC_ChannelConfTypeDef sConfig;

/**
 * @}
 */

uint32_t I2C_SHIELDS_Timeout = NUCLEO_I2C_SHIELDS_TIMEOUT_MAX;    /*<! Value of Timeout when I2C communication fails */ 

static I2C_HandleTypeDef    I2C_SHIELDS_Handle;

//osMutexId I2C1_Mutex_id = 0;

/** @defgroup NUCLEO_F401RE_LOW_LEVEL_Private_FunctionPrototypes
 * @{
 */

/* Link function for PRESSURE peripheral */
void            PRESSURE_IO_Init(void);
void            PRESSURE_IO_ITConfig(void);
void            PRESSURE_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t WriteAddr, uint16_t NumByteToWrite);
void            PRESSURE_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead);

/* Link function for UV peripheral */
void            UV_IO_Init(void);
void            UV_IO_ITConfig(void);
void            UV_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t WriteAddr, uint16_t NumByteToWrite);
void            UV_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead);

/* Link function for HUM_TEMP peripheral */
void            HUM_TEMP_IO_Init(void);
void            HUM_TEMP_IO_ITConfig(void);
void            HUM_TEMP_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t WriteAddr, uint16_t NumByteToWrite);
void            HUM_TEMP_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead);


static void I2C_SHIELDS_MspInit(void);
static void I2C_SHIELDS_Error(uint8_t Addr);

/**
 * @}
 */

/** @defgroup NUCLEO_F401RE_LOW_LEVEL_Private_Functions
 * @{
 */

/**
 * @brief  This method returns the STM32L0XX NUCLEO BSP Driver revision
 * @param  None
 * @retval version : 0xXYZR (8bits for each decimal, R for RC)
 */
uint32_t BSP_GetVersion(void)
{
    return __NUCLEO_F401RE_BSP_VERSION;
}


/**
 * @brief  Configures Button GPIO and EXTI Line.
 * @param  Button: Specifies the Button to be configured.
 *   This parameter should be: BUTTON_KEY
 * @param  ButtonMode: Specifies Button mode.
 *   This parameter can be one of following parameters:
 *     @arg  BUTTON_MODE_GPIO: Button will be used as simple IO
 *     @arg BUTTON_MODE_EXTI: Button will be connected to EXTI line with interrupt
 *                            generation capability
 * @retval None
 */
void BSP_PB_Init(Button_TypeDef Button, ButtonMode_TypeDef Button_Mode)
{
    GPIO_InitTypeDef GPIO_InitStruct;

    /* Enable the BUTTON Clock */
    BUTTONx_GPIO_CLK_ENABLE(Button);
    __SYSCFG_CLK_ENABLE();

    if(Button_Mode == BUTTON_MODE_GPIO)
    {
        /* Configure Button pin as input */
        GPIO_InitStruct.Pin = BUTTON_PIN[Button];
        GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
        GPIO_InitStruct.Pull = GPIO_NOPULL;
        GPIO_InitStruct.Speed = GPIO_SPEED_FAST;

        HAL_GPIO_Init(BUTTON_PORT[Button], &GPIO_InitStruct);
    }

    if(Button_Mode == BUTTON_MODE_EXTI)
    {
        /* Configure Button pin as input with External interrupt */
        GPIO_InitStruct.Pin = BUTTON_PIN[Button];
        GPIO_InitStruct.Pull = GPIO_NOPULL;
        GPIO_InitStruct.Speed = GPIO_SPEED_FAST;
        GPIO_InitStruct.Mode = GPIO_MODE_IT_FALLING;
        HAL_GPIO_Init(BUTTON_PORT[Button], &GPIO_InitStruct);

        /* Enable and set Button EXTI Interrupt to the lowest priority */
        NVIC_SetPriority((IRQn_Type)(BUTTON_IRQn[Button]), 0x03);
        HAL_NVIC_EnableIRQ((IRQn_Type)(BUTTON_IRQn[Button]));
    }
}

/**
 * @brief  Returns the selected Button state.
 * @param  Button: Specifies the Button to be checked.
 *   This parameter should be: BUTTON_KEY
 * @retval The Button GPIO pin value.
 */
uint32_t BSP_PB_GetState(Button_TypeDef Button)
{
    return HAL_GPIO_ReadPin(BUTTON_PORT[Button], BUTTON_PIN[Button]);
}

/******************************* I2C Routines**********************************/
/**
 * @brief  Configures I2C interface.
 * @param  None
 * @retval None
 */
static void I2C_SHIELDS_Init(void)
{
    if(HAL_I2C_GetState(&I2C_SHIELDS_Handle) == HAL_I2C_STATE_RESET)
    {
        /* I2C_SHIELDS peripheral configuration */
        I2C_SHIELDS_Handle.Init.ClockSpeed = NUCLEO_I2C_SHIELDS_SPEED;
        I2C_SHIELDS_Handle.Init.DutyCycle = I2C_DUTYCYCLE_2;
        I2C_SHIELDS_Handle.Init.OwnAddress1 = 0x33;
        I2C_SHIELDS_Handle.Init.AddressingMode = I2C_ADDRESSINGMODE_7BIT;
        I2C_SHIELDS_Handle.Instance = NUCLEO_I2C_SHIELDS;

        /* Init the I2C */
        I2C_SHIELDS_MspInit();
        HAL_I2C_Init(&I2C_SHIELDS_Handle);
    }
}

/**
 * @brief  Write a value in a register of the device through BUS.
 * @param  Addr: Device address on BUS Bus.
 * @param  Reg: The target register address to write
 * @param  Value: The target register value to be written
 * @retval HAL status
 */

static void I2C_SHIELDS_WriteData(uint8_t* pBuffer, uint8_t Addr, uint8_t Reg, uint16_t Size)
{
    HAL_StatusTypeDef status = HAL_OK;

    status = HAL_I2C_Mem_Write(&I2C_SHIELDS_Handle, Addr, (uint16_t)Reg, I2C_MEMADD_SIZE_8BIT, pBuffer, Size, I2C_SHIELDS_Timeout);

    /* Check the communication status */
    if(status != HAL_OK)
    {
        /* Execute user timeout callback */
        I2C_SHIELDS_Error(Addr);
    }
}

/**
 * @brief  Read a register of the device through BUS
 * @param  Addr: Device address on BUS .
 * @param  Reg: The target register address to read
 * @retval HAL status
 */
static void I2C_SHIELDS_ReadData(uint8_t* pBuffer, uint8_t Addr, uint8_t Reg, uint16_t Size)
{
    HAL_StatusTypeDef status = HAL_OK;

    status = HAL_I2C_Mem_Read(&I2C_SHIELDS_Handle, Addr, (uint16_t)Reg, I2C_MEMADD_SIZE_8BIT, pBuffer, Size, I2C_SHIELDS_Timeout);

    /* Check the communication status */
    if(status != HAL_OK)
    {
        /* Execute user timeout callback */
        I2C_SHIELDS_Error(Addr);
    }

}

/**
 * @brief  Manages error callback by re-initializing I2C.
 * @param  Addr: I2C Address
 * @retval None
 */
static void I2C_SHIELDS_Error(uint8_t Addr)
{
    /* De-initialize the I2C comunication bus */
    HAL_I2C_DeInit(&I2C_SHIELDS_Handle);

    /* Re-Initiaize the I2C comunication bus */
    I2C_SHIELDS_Init();
}

/**
 * @brief I2C MSP Initialization
 * @param None
 * @retval None
 */
static void I2C_SHIELDS_MspInit(void)
{
    GPIO_InitTypeDef  GPIO_InitStruct;

    /*if(!NUCLEO_I2C_SHIELDS_MUTEX) {
        NUCLEO_I2C_SHIELDS_MUTEX = osMutexCreate(0);
    }*/
    /* Enable I2C GPIO clocks */
    NUCLEO_I2C_SHIELDS_SCL_SDA_GPIO_CLK_ENABLE();

    /* I2C_SHIELDS SCL and SDA pins configuration -------------------------------------*/
    GPIO_InitStruct.Pin = NUCLEO_I2C_SHIELDS_SCL_PIN | NUCLEO_I2C_SHIELDS_SDA_PIN;
    GPIO_InitStruct.Mode = GPIO_MODE_AF_OD;
    GPIO_InitStruct.Speed = GPIO_SPEED_FAST;
    GPIO_InitStruct.Pull  = GPIO_NOPULL;
    GPIO_InitStruct.Alternate  = NUCLEO_I2C_SHIELDS_SCL_SDA_AF;
    HAL_GPIO_Init(NUCLEO_I2C_SHIELDS_SCL_SDA_GPIO_PORT, &GPIO_InitStruct);

    /* Enable the I2C_SHIELDS peripheral clock */
    NUCLEO_I2C_SHIELDS_CLK_ENABLE();

    /* Force the I2C peripheral clock reset */
    NUCLEO_I2C_SHIELDS_FORCE_RESET();

    /* Release the I2C peripheral clock reset */
    NUCLEO_I2C_SHIELDS_RELEASE_RESET();

    /* Enable and set I2C_SHIELDS Interrupt to the highest priority */
    HAL_NVIC_SetPriority(NUCLEO_I2C_SHIELDS_EV_IRQn, 0, 0);
    HAL_NVIC_EnableIRQ(NUCLEO_I2C_SHIELDS_EV_IRQn);

    /* Enable and set I2C_SHIELDS Interrupt to the highest priority */
    HAL_NVIC_SetPriority(NUCLEO_I2C_SHIELDS_ER_IRQn, 0, 0);
    HAL_NVIC_EnableIRQ(NUCLEO_I2C_SHIELDS_ER_IRQn);
}


/********************************* LINK PRESSURE *****************************/
/**
 * @brief  Configures Pressure I2C interface.
 * @param  None
 * @retval None
 */
void PRESSURE_IO_Init(void)
{
    I2C_SHIELDS_Init();
}

/**
 * @brief  Writes a buffer to the Pressure sensor.
 * @param  pBuffer: pointer to data to be written.
 * @param  DeviceAddr: specifies the slave address to be programmed.
 * @param  RegisterAddr: specifies the Pressure register to be written.
 * @param  NumByteToWrite: number of bytes to be written.
 * @retval None.
 */
void PRESSURE_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToWrite)
{
    //NUCLEO_I2C_SHIELDS_MUTEX_TAKE();
    /* call I2C_SHIELDS Read data bus function */
    I2C_SHIELDS_WriteData(pBuffer, DeviceAddr, RegisterAddr, NumByteToWrite);
    //NUCLEO_I2C_SHIELDS_MUTEX_RELEASE();
}

/**
 * @brief  Reads a buffer from the Pressure sensor.
 * @param  pBuffer: pointer to data to be read.
 * @param  DeviceAddr: specifies the address of the device.
 * @param  RegisterAddr: specifies the Pressure internal address register to read from.
 * @param  NumByteToRead: number of bytes to be read.
 * @retval None.
 */
void PRESSURE_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead)
{
    //NUCLEO_I2C_SHIELDS_MUTEX_TAKE();
    /* call I2C_SHIELDS Read data bus function */
    I2C_SHIELDS_ReadData(pBuffer, DeviceAddr, RegisterAddr, NumByteToRead);
    //NUCLEO_I2C_SHIELDS_MUTEX_RELEASE();
}




/********************************* LINK UVI *****************************/
/**
 * @brief  Configures Uvi I2C interface.
 * @param  None
 * @retval None
 */
void UV_IO_Init(void)
{
    I2C_SHIELDS_Init();
}

/**
 * @brief  Writes a buffer to the Uvi sensor.
 * @param  pBuffer: pointer to data to be written.
 * @param  DeviceAddr: specifies the slave address to be programmed.
 * @param  RegisterAddr: specifies the Pressure register to be written.
 * @param  NumByteToWrite: number of bytes to be written.
 * @retval None.
 */
void UV_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToWrite)
{
    //NUCLEO_I2C_SHIELDS_MUTEX_TAKE();
    /* call I2C_SHIELDS Read data bus function */
    I2C_SHIELDS_WriteData(pBuffer, DeviceAddr, RegisterAddr, NumByteToWrite);
    //NUCLEO_I2C_SHIELDS_MUTEX_RELEASE();
}

/**
 * @brief  Reads a buffer from the Uvi sensor.
 * @param  pBuffer: pointer to data to be read.
 * @param  DeviceAddr: specifies the address of the device.
 * @param  RegisterAddr: specifies the Pressure internal address register to read from.
 * @param  NumByteToRead: number of bytes to be read.
 * @retval None.
 */
void UV_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead)
{
    //NUCLEO_I2C_SHIELDS_MUTEX_TAKE();
    /* call I2C_SHIELDS Read data bus function */
    I2C_SHIELDS_ReadData(pBuffer, DeviceAddr, RegisterAddr, NumByteToRead);
    //NUCLEO_I2C_SHIELDS_MUTEX_RELEASE();
}





/********************************* LINK HUM_TEMP *****************************/
/**
 * @brief  Configures Uvi I2C interface.
 * @param  None
 * @retval None
 */
void HUM_TEMP_IO_Init(void)
{
    I2C_SHIELDS_Init();
}

/**
 * @brief  Writes a buffer to the Uvi sensor.
 * @param  pBuffer: pointer to data to be written.
 * @param  DeviceAddr: specifies the slave address to be programmed.
 * @param  RegisterAddr: specifies the Pressure register to be written.
 * @param  NumByteToWrite: number of bytes to be written.
 * @retval None.
 */
void HUM_TEMP_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToWrite)
{
    //NUCLEO_I2C_SHIELDS_MUTEX_TAKE();
    /* call I2C_SHIELDS Read data bus function */
    I2C_SHIELDS_WriteData(pBuffer, DeviceAddr, RegisterAddr, NumByteToWrite);
    //NUCLEO_I2C_SHIELDS_MUTEX_RELEASE();
}

/**
 * @brief  Reads a buffer from the Uvi sensor.
 * @param  pBuffer: pointer to data to be read.
 * @param  DeviceAddr: specifies the address of the device.
 * @param  RegisterAddr: specifies the Pressure internal address register to read from.
 * @param  NumByteToRead: number of bytes to be read.
 * @retval None.
 */
void HUM_TEMP_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead)
{
    //NUCLEO_I2C_SHIELDS_MUTEX_TAKE();
    /* call I2C_SHIELDS Read data bus function */
    I2C_SHIELDS_ReadData(pBuffer, DeviceAddr, RegisterAddr, NumByteToRead);
    //NUCLEO_I2C_SHIELDS_MUTEX_RELEASE();
}

/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/

