/**
  ******************************************************************************
  * @file    uvis3.h
  * @author  AST Robotics Team
  * @version V0.0.1
  * @date    18-February-2014
  * @brief   This file contains definitions uvis3.c 
  *          firmware driver.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2014 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __UVIS3_H
#define __UVIS3_H

#ifdef __cplusplus
 extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "../Common/uv.h"

/** @addtogroup BSP
  * @{
  */

/** @addtogroup UVIS3
  * @{
  */
  
/** @defgroup UVIS3_Exported_Constants
  * @{
  */

/******************************************************************************/
/*************************** START REGISTER MAPPING  **************************/
/******************************************************************************/
/**
 * @brief Device Address
 */
#define UVIS3_ADDRESS                                      0x8E


/**
 * @brief Device identifier register.
 * \code
 * Read
 * Default value: 0xBD
 * 7:0 This read-only register contains the device identifier that, 
       for UVIS3, is set to 0xCA.
 * \endcode
 */
#define UVIS3_WHO_AM_I_ADDR                             0x0F
  

 /**
 * @brief UVIS3 control register 1
 * \code
 * Read/write
 * Default value: 0x80.
 * 7 PD: power down control. 1 - disable; 0 - enable.
 * 6:2 reserved.
 * 1 BDU: Block Data Update. 0 - continuous update; 1 -output registers not 
          updated until MSB and LSB reading. //CHECK
 * 0 ODR: Output Data Rate. 0 - ODR disabled (one shot mode); 1 ODR at 1 Hz.
 * \endcode
 */
#define UVIS3_CTRL_REG1_ADDR                            0x20


 /**
 * @brief UVIS3 control register 2
 * \code
 * Read/write
 * Default value: 0x00.
 * 7 BOOT: reboot memory content. 1 - reload flash content; 0 - normal mode.
 * 6:5 reserved.
 * 4: I2C_DIS: Disable I2C interface. 0: enable; 1: disable.
 * 3 SIM: SPI Serial Interface Mode Selection. 0: 4 wires interface; 
          1: 3 wires interface.
 * 2:1 Reserved
 * 0: One shot enable. 0: waiting for start of conversion; 
      1: start for a new dataset.
 * \endcode
 */
#define UVIS3_CTRL_REG2_ADDR                            0x21


 /**
 * @brief UVIS3 control register 3
 * \code
 * Read/write
 * Default value: 0x00.
 * 7 INT_H_L: Interrupt active high, low. 0: active high; 1: active low.
 * 6 PP_OD: Push-pull/open drain selection on interrupt pads. 0: push-pull; 
            1: open drain.
 * 5:2 Reserved
 * 1:0 INT1_S2, INT1_S1: data signal on INT1 pad control bits. 
                         Default value: 00.
 *   |  INT1_S2  | INT1_S1  | INT1 pin               |
 *   -------------------------------------------------
 *   |     0     |  0       |  Data ready            |        
 *   |     0     |  1       |  UV index High         |                           
 *   |     1     |  0       |  UV index Low          |                            
 *   |     1     |  1       |  UV index High or Low  |                 
 * \endcode
 */
#define UVIS3_CTRL_REG3_ADDR                            0x22


 /**
 * @brief INT1 interrupt configuration
 * \code
 * Read/write
 * Default value: 0x00.
 * 7:4 Reserved.
 * 3: DIFF_EN: interrupt logical block enable. 0: interrupt logick block 
               disable; 1: enable.
 * 2: LIR: Latch interrupt request into INT_SOURCE register with (24h), with 
           the INT_SOURCE register cleared by reading INT_SOURCE reg (24h). 
           0: interrupt request not latched;1: latched.
 * 1: UVLE: Enable interrupt generation on differential UV low event. 0: disable 
            interrupt request;1: enable interrupt request
 * 0: UVHE: Enable interrupt generation on differential UV high event SIM; 
            0: disable interrupt request;1: enable interrupt request
 * \endcode
 */
#define UVIS3_INT_CFG_ADDR                              0x23



 /**
 * @brief Interrupt source configuration register
 * \code
 * Read/write
 * Default value: 0x00.
 * 7:3 Reserved.
 * 2: IA: Interrupt Active. 0: no interrupt has been generated; 
          1: one or more interrupt events have been generated. 
 * 1: UVL: Differential UV low. 0: no interrupt; 1: UVL event has occurred.
 * 0: UVH: Differential UV high. 0: no interrupt; 1: UVH event has occurred.
 * \endcode
 */
#define UVIS3_INT_SRC_ADDR                              0x24



 /**
 * @brief Differential UV interrupt threshold
 * \code
 * Read/write
 * Default value: 0x00.
 * 7:0 Differential UV Interrupt Threshold values.
 * \endcode
 */
#define UVIS3_DIFF_UV_TSH_ADDR                          0x25


 /**
 * @brief Status register
 * \code
 * Read/write
 * Default value: 0x00.
 * 7:1 Reserved.
 * 0: UV_DA: UV data available. 0: UV data index not available; 1: UV data 
             index available.
 * \endcode
 */
#define UVIS3_STATUS_REG_ADDR                           0x27


 /**
 * @brief UV index data
 * \code
 * Read/write
 * Default value: 0x00.
 * 7:0 UV index data output values.
 * \endcode
 */
#define UVIS3_UVDATA_OUT_ADDR                           0x28


/******************************************************************************/
/**************************** END REGISTER MAPPING  ***************************/
/******************************************************************************/

/**
 * @brief Device Identifier. Default value of the WHO_AM_I register.
 */
#define I_AM_UVIS3                 ((uint8_t)0xCA)

/** @defgroup Power_Mode_selection CTRL_REG1
  * @{
  */
#define UVIS3_MODE_ACTIVE          		((uint8_t)0x00)
#define UVIS3_MODE_POWERDOWN       		((uint8_t)0x80)

#define UVIS3_MODE_MASK                 ((uint8_t)0x80)

/**
  * @}
  */ 

/** @defgroup OutPut_DataRate_Selection CTRL_REG1
  * @{
  */
#define UVIS3_ODR_ONE_SHOT              ((uint8_t)0x00)
#define UVIS3_ODR_1HZ    	            ((uint8_t)0x01)

#define UVIS3_ODR_MASK                  ((uint8_t)0x01)

/**
  * @}
  */

/** @defgroup Boot_Mode_selection CTRL_REG2
  * @{
  */
#define UVIS3_BOOT_NORMALMODE           ((uint8_t)0x00)
#define UVIS3_BOOT_REBOOTMEMORY         ((uint8_t)0x80)

#define UVIS3_BOOT_MASK                 ((uint8_t)0x80)

/**
  * @}
  */


/** @defgroup UVIS3_Exported_Functions
  * @{
  */
/* Sensor Configuration Functions */ 
void      UVIS3_Init(UV_InitTypeDef *UVIS3_Init);
uint8_t   UVIS3_ReadID(void);
void      UVIS3_RebootCmd(void);
void      UVIS3_GetIndex(float* pfData);

/* Interrupt Configuration Functions */
void      UVIS3_INT1InterruptConfig(uint16_t Int1Config);
void      UVIS3_EnableIT(void);
void      UVIS3_DisableIT(void);

/* Uv sensor driver structure */
extern UV_DrvTypeDef Uvis3Drv;

/* Uv sensor IO functions */
void      UV_IO_Init(void);
void      UV_IO_DeInit(void);
void      UV_IO_Write(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t WriteAddr, uint16_t NumByteToWrite);
void      UV_IO_Read(uint8_t* pBuffer, uint8_t DeviceAddr, uint8_t RegisterAddr, uint16_t NumByteToRead);

/**
  * @}
  */

/**
  * @}
  */ 
  
/**
  * @}
  */ 
  
#ifdef __cplusplus
  }
#endif
  
#endif /* __UVIS3_H */


/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 

