/* MMEx for MBED - SPI interfacing and ringbuffer functions
 * Copyright (c) 2011 MK
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
  \file spissp.h
  \brief SPI interfacing and ringbuffer functions
*/

#ifndef SPISSP_H
#define SPISSP_H

#include "mbed.h"
#include "lpc17xx_ssp.h"
#include "globaldefs.h"

// macros for managing the SSP interface SSP0 only!
#define tx_fifo_notfull  (LPC_SSP0->SR & SSP_STAT_TXFIFO_NOTFULL)
#define tx_fifo_empty    (LPC_SSP0->SR & SSP_STAT_TXFIFO_EMPTY)
#define rx_fifo_notempty (LPC_SSP0->SR & SSP_STAT_RXFIFO_NOTEMPTY)
#define rx_fifo_full     (LPC_SSP0->SR & SSP_STAT_RXFIFO_FULL)

#define irq_RTMIS        (LPC_SSP0->MIS & SSP_MIS_RT)    // RX FIFO timeout
#define irq_RXMIS        (LPC_SSP0->MIS & SSP_MIS_RX)    // RX FIFO half full
#define irq_RISOVR       (LPC_SSP0->RIS & SSP_RIS_ROR)   // receive overrun
 
#define DBG_OFF  0
#define DBG_ON   1
#define DBG_FULL 2 
 

 /** Library for the interrupt driven SPI SLave
 *
 */
class spislirq {

public:

 /** initialize the SPI interface
 *
 */ 
  spislirq(int PortNum);

/** initialization of SPI Slave and buffers
 *  
 */  
  void init();
  
/** attach callback for network (or other?) polling
 *  
 */  
  void attach(void (*function)(void)) {
    _callback.attach(function);  
  }
   /** attach callback for network (or other?) polling
 *  
 */  
   
/** disable SSP0 interrupts
 *  will discard the byte if buffer full
 */  
  void disable_irq(void);
  
/** enable SSP0 interrupts
 *  will discard the byte if buffer full
 */  
  void enable_irq(void);

/** add one byte to the transmit buffer, no blocking, no protocol
 *  will discard the byte if buffer full
 *
 *  @param c byte to send
 */  
  void txx_add(unsigned char c);

/** returns the number of bytes available in the transmit buffer
 *
 *  @return available space in transmit buffer
 */
  int tx_room();
  
/** returns the amount af actual bytes used in the transmit buffer
 *
 *  @return bytes used in transmit buffer
 */  
  int tx_use();
  
/** returns the number of bytes available in the receive buffer
 *
 *  @return available space in transmit buffer
 */  
  int rx_room();
  
/** returns the percentage used in the transmit buffer
 *
 *  @return percentage from 0 to 100
 */  
  int tx_perc();
  
/** returns the percentage used in the receive buffer
 *
 *  @return percentage from 0 to 100
 */  
  int rx_perc();
  
/** returns the amount af actual bytes used in the receive buffer
 *
 *  @return bytes used in receive buffer
 */  
  int rx_use();
  
/** empties the transmit buffer
 */  
  void flush_tx();
  
/** empties the receive buffer
 */  
  void flush_rx();
  
/** add one byte to the transmit buffer, no protocol handling is done
 *
 *  @param Bt byte to send
 *  @return 0 when succesfull, will block until there is room in the buffer
 *
 */  
  int tx_add(unsigned char Bt);
  
/** add one byte to the transmit buffer with protocol handling,
 *  function will block until there is room in the transmit buffer
 *
 *  @param Bt byte to send
 */  
  void tx_addp(unsigned char Bt);
  
/** adds a complete string to the transmit buffer, with protocol processing.
 *  function will block until there is room in the transmit buffer
 *
 *  @param S string to be transmitted
 *  @return always returns 0
 */  
  int tx_string(char S[]);
  
/** read one byte from the receive buffer without protocol processing
 *
 *  @return the character read, blocks if RX buffer is empty!
 *
 */  
  int rx_read();
  
/** inserts one byte in the receive buffer without protocol processing
 *  used for command batch processing
 *
 *  @return the character read, blocks if RX buffer is empty!
 *
 */  
  int rx_add(unsigned char c);
  
/** peeks in receive buffer, reads next pending input char without advancing buffer pointer
 *
 *  @return the character read, blocks if RX buffer is empty!
 *
 */  
  int rx_peek();
  
/** read one byte, will wait until a valid char is read,
 *  this function is blocking and will process escape characters,
 *  when an escape is seen, will try to read the next char
 *
 *  @param mode command line processing or data read (ignored)
 *  @return character read, or -1 when >F seen, -2 when >I seen
 *
 */  
  int rxx_read(bool mode);  
  void DBG_set(int l);

/** check if the TX buffer is empty
 * @return true if the TX buffer is empty
 */
  bool tx_empty();
  
/** check if the TX buffer is full
 * @return true if the TX buffer is full
 */  
  bool tx_full();
  
/** check if the RX buffer is empty
 * @return true if the RX buffer is empty
 */  
  bool rx_empty();
  
/** check if the RX buffer is full
 * @return true if the RX buffer is full
 */  
  bool rx_full();


private:

SPISlave smldl;        // uses SPI Class from MBED library
Serial pc;
DigitalOut _led4;
FunctionPointer _callback;

int hex2int(char C);
void spi_isr();        // interrupt service routine
static void _spi_isr(void);
static spislirq *instance;
void DBG_outchar(char C);
void DBG_inchar(char C);
void DBG_chr(char* src, char c);
void do_callback(void) { _callback.call(); };
  

// buffers and pointers in buffers
#define bufsize   255
unsigned char rxxbuf[bufsize];     // receive buffer
unsigned char txxbuf[bufsize];     // transmit buffer
int tx_in;       
int tx_out;        
int rx_in;
int rx_out;

int DBG_level;

};

#endif