/* Driver for brushed DC Motor Controler */
/* ************************************* */



/* Include Files Here */
#include "mbed.h"                                /* mbed header file */
#include "misra_types.h"                         /* MISRA Types header file */
#include "defines.h"
#include "mbed_Port_Structure.h"                /* Port structure for MBED Module */





    /* 
    The standard software module for the MC1 has been modified to accomodate the mapping of the mbed pins to it.
    Not all of the functions usually available for control of the MC1 module are available.
    Below in the commented out section is the full section of functions.
    In the function prototype section are the ones that are implemented 
    */

    /*
    The mapping of the various functions to the mbed module are as follows...
    int16_t RSEDP_MC1_Read_VDCLINK(uint8_t channel);        VDCLINK - Analogue AN2 - pin 17 mbed MCU            
    uint16_t RSEDP_MC1_Read_VSENSE(uint8_t channel);        VSENSE - Analogue AN0 - pin 15 mbed MCU        
    uint16_t RSEDP_MC1_Read_VTACH(uint8-t channel);            VTACH - possible Analogue AN4 - pin 19 mbed MCU - but not implemnted as this pin 19 used as brake input
    uint16_t RSEDP_MC1_Read_Acceleration_Ramp(void);        Accelration ramp - Not implemented        
    uint16_t RSEDP_MC1_Read_Deceleration_Ramp(void);        Deceleration lamp - not implemented        


    uint8_t RSEDP_MC1_Read_External_Fault_Input(void);                        External Fault - not implemented        
    uint8_t RSEDP_MC1_Read_Fault_Reset_Switch(void);                        Fault Reset - not implemented
    uint8_t RSEDP_MC1_Read_Open_Switch(uint8_t pin_option);                    Open Switch - Mapped to EVM2_GPIO41 - pin 11 mbed digital input        
    uint8_t RSEDP_MC1_Read_Closed_Switch(uint8_t pin_option);                Closed Switch - not implemented         
    uint8_t RSEDP_MC1_Read_Encoder0_Input(uint8_t pin_option);                Read Encoder0 - mapped to EVG6_GPIO52 - pin 30 mbed digital input (possible timer input capture)                 
    uint8_t RSEDP_MC1_Read_Encoder1_Input(uint8_t pin_option);                Read Encoder1 - mapped to EVG7_GPIO54 - pin    8 mbed digital input
    uint8_t RSEDP_MC1_Read_Tacho_Input(uint8_t pin_option);                    Tacho Pulses - Duplicate function to Encoder0         
    uint8_t RSEDP_MC1_Read_Motor_RunStop(uint8_t pin_option);                Motor Run Stop - mapped to EVG3_GPIO46 pin 26 mbed digital input
    uint8_t RSEDP_MC1_Read_Motor_Direction(uint8_t pin_option);                Read Motor Direction - not implemented
    uint8_t RSEDP_MC1_Read_Current_Limit_Match(uint8_t pin_option);            Motor Current Limit - not implemented


    void RSEDP_MC1_Set_Motor_Direction(uint8_t motor_direction, uint8_t pin_option);            Set Motor Direction - mapped to pin 24 - digital out
    void RSEDP_MC1_Brake(uint8_t brake_onoff, uint8_t pin_option);                                Brake on/off - mapped to GPIO0 - pin 19 mbed module digital out
    void RSEDP_MC1_Set_Motor_Speed_Duty(float motor_speed_duty, uint8_t pin_option);            Set motor speed duty - mapped to EVG0_GPIO40 - pin 23 mbed PWM out
    void RSEDP_MC1_Set_Current_Trip_Duty(float current_trip_duty, uint8_t pin_option);            Set current trip duty - mapped to CPU_DACO0_GPIO17 - pin 21 PWM out
    */

    
    



/* function Prototypes */
void RSEDP_MC1_Setup(void);                                        /* Setup the MC1 Board */

/* Analogue Levels */
uint16_t RSEDP_MC1_Read_VDCLINK(uint8_t channel);                /* Read the filtered 12V DC voltage on the MC1 board via pot down resistors */
uint16_t RSEDP_MC1_Read_VSENSE(uint8_t channel);                /* Read the filtered current sense output from the MC1 board */


/* Digital level inputs */
uint8_t RSEDP_MC1_Read_Encoder0_Input(uint8_t pin_option);        /* Read the physical level on the Encoder0 input pin on P301 pin 8 on the MC1 board*/
uint8_t RSEDP_MC1_Read_Encoder1_Input(uint8_t pin_option);        /* Read the physical level on the Encoder1 input pin on P301 pin 10 on the MC1 board*/
uint8_t RSEDP_MC1_Read_Tacho_Input(uint8_t pin_option);            /* Read the physical level on the Tacho Pulses Input pin on P301 pin 6 on the MC1 board */


/* Outputs */
void RSEDP_MC1_Set_Motor_Direction(uint8_t motor_direction, uint8_t pin_option);            /* Set the motor direction */
void RSEDP_MC1_Brake(uint8_t brake_onoff, uint8_t pin_option);                                /* turn the brake on or off */
void RSEDP_MC1_Set_Motor_Speed_Duty(float motor_speed_duty, uint8_t pin_option);            /* Power control of motor speed controller using PWM */






/* Setup the MC1 Board */
void RSEDP_MC1_Setup(void)
    {
        PwmOut_Pin23.period_us(50);                                        /* 20KHz fundemental PWM */
        RSEDP_MC1_Set_Motor_Speed_Duty(0, EVG0_GPIO40);                    /* Speed set to zero */
        RSEDP_MC1_Brake(BRAKE_ON, GPIO0);                                /* Brake ON */                    
    }


/* Read The VDCLINK Voltage */
uint16_t RSEDP_MC1_Read_VDCLINK(uint8_t channel)
    {
        /* There are two possible AD conveter channels where we can measure the VDCLINK voltage */
        /* Either ADC channel AN2 or ADC channel AN12 selectable via JP312 */
        /* channel therefore has two value 2 or 12 */

        float return_value = 0;

        if (channel == AN2)
            {
                /* Link JP312 1-2 */
                return_value = (AD2 * 3300);
            }
            
        if (channel == AN12)
            {
                /* Link JP312 2-3 */
                /* AN12 is mapped onto AD3 on the mbed module via link option */
                return_value = (AD3 * 3300);
            }
        
        return (uint16_t) return_value;
    }
    
    
/* Read the current sense output from the MC1 board */    
uint16_t RSEDP_MC1_Read_VSENSE(uint8_t channel)    
    {
        /* Read the current sense output from the motor drive device */
        /* This voltage signal is proprtional to the motor current */
        /* We can read this from a choice of two AD channels AN0 and AN10 selectable via JP307 */
        
        float return_value = 0;
        
        if (channel == AN0)
            {
                /* Link JP307 1-2 */
                return_value = (AD0 * 3300);
                
            }
            
        if (channel == AN10)
            {
                /* Link JP307 2-3 */
                /* AN10 is mapped onto AD1 via link option on the mbed module */
                return_value = (AD1 * 3300);
            }

        return (uint16_t) return_value;                                    
    }



/* Read the filtered tacho voltage. This voltage is proportional to RPM */
uint16_t RSEDP_MC1_Read_VTACH(uint8_t channel)
    {
        /* The VTACH signal is an analogue voltage produces by the tacho which is proprtional to RPM. The faster the rotation the higher the votlage */
        /* Not all tachos produce a variable output voltage but the motor drive module is able to cope with both pulse fed tacho and also voltage supplied tacho voltage feedback signals */
        /* This function is for the feedback voltage method */
        /* The VTACH voltage is available on two analogue pins AN4 and AN14 selectable via JP304 */

        if (channel == AN4)
            {
                /* Link JP304 1-2 */
                /* not implemented */
            }
            
        if (channel == AN14)
            {
                /* Link JP304 2-3 */
                /* not implemented - AN14 is not mapped on to the MBED module */                
            }

        /* not implemented */
        return 0;
    }






    
/* Read the physical level on the Encoder0 input pin on P301 pin 8 on the MC1 board*/
uint8_t RSEDP_MC1_Read_Encoder0_Input(uint8_t pin_option)        
    {
        /* The Encoder0 pin can be routed to two output pins EVM5_GPIO47 or EVG6_GPIO52 via JP302 on the MC1 motor Drive module */
        /* To use this Encoder input JP303 also needs to be set in the 2-3 position */

        uint8_t return_value = 0;
        
        if (pin_option == EVM5_GPIO47)
            {
                /* Links JP302 1-2 and JP303 2-3 */
                /* not implemented */    
            }
            
        if (pin_option == EVG6_GPIO52)
            {
                /* Links JP302 2-3 and JP303 2-3 */    
                return_value = DigitalIn_Pin30;
            }

        return return_value;
    }



/* Read the physical level on the Encoder1 input pin on P301 pin 10 on the MC1 board*/
uint8_t RSEDP_MC1_Read_Encoder1_Input(uint8_t pin_option)        
    {
        /* The Encoder1 pin can be routed to two output pins EVM6_GPIO49 or EVG7_GPIO54 via JP306 on the MC1 motor Drive module */
        
        uint8_t return_value = 0;
                
        if (pin_option == EVM6_GPIO49)
            {
                /* Link option JP306 1-2 */
                /* not implemented */
            }
            
        if (pin_option == EVG7_GPIO54)
            {
                /* Link option JP306 2-3 */
                return_value = DigitalIn_Pin8;
            }

        return return_value;
    }



/* Read the physical level on the Tacho Pulses Input pin on P301 pin 6 on the MC1 board */
uint8_t RSEDP_MC1_Read_Tacho_Input(uint8_t pin_option)            
    {
        /* The Tacho Pulses pin uses the same hardware as the Encoder0 pin and either input is selected via jumper JP303 */
        /* This tacho pulse can like the Encoder0 pin be routed to two output pins EVM5_GPIO47 or EVG6_GPIO52 */
        /* to use this Encoder input JP303 needs to be set in the 1-2 position */
        
        return RSEDP_MC1_Read_Encoder0_Input(pin_option);
    }
    






/* Set the motor direction */            
void RSEDP_MC1_Set_Motor_Direction(uint8_t motor_direction, uint8_t pin_option)
    {
        /* the Motor Direction can be set by the MCU via EVG1_GPIO42 or EVG5_GPIO50 selectable via link option JP206 and JP207 on MC1 motor drive module */
        if (pin_option == EVG1_GPIO42)
            {
                /* Link options JP206 1-2 and JP207 2-3 */
                if (motor_direction == FORWARD)
                    {
                        DigitalOut_Pin24 = 1;
                    }
                else{
                        DigitalOut_Pin24 = 0;
                    }
            }
            
        if (pin_option == EVG5_GPIO50)
            {
                /* Link options JP206 2-3 and JP207 2-3 */
                /* Not implemented */
            }
    }
    
    

/* turn the brake on or off */    
void RSEDP_MC1_Brake(uint8_t brake_onoff, uint8_t pin_option)
    {
        /* The brake can be turned ON or OFF via this function call */
        /* A choice of two pins can beused to control the brake, GPIO0 or GPIO1 selectable via jumper JP209 */
        if (pin_option == GPIO0)
            {
                /* Link option JP209 2-3 */
                if (brake_onoff == BRAKE_ON)
                    {
                        DigitalOut_Pin19 = 1;
                    }
                else{
                        DigitalOut_Pin19 = 0;
                    }
            }
            
        if (pin_option == GPIO1)
            {
                /* Link option JP209 1-2 */
                /* Not implmented */
            }
    }                                                
    

    
/* Power control of motor speed controller using PWM */
void RSEDP_MC1_Set_Motor_Speed_Duty(float motor_speed_duty, uint8_t pin_option)
    {
        /* This function controls the speed of the motor by controlling the PWM duty of the bridge */
        /* The duty is fed by a PWM signal either from EVG0_GPIO40 or EVG4_GPIO48 selectable via link option JP204 and link JP203 & JP205 */
        if (pin_option == EVG0_GPIO40)
            {
                /* Link option JP204 1-2 and Link JP203 2-3 and JP205 1-2 */
                PwmOut_Pin23 = (1.0f - motor_speed_duty);
            }
            
        if (pin_option == EVG4_GPIO48)
            {
                /* Link option JP204 2-3 and Link JP203 2-3 and JP205 1-2  */
                /* not implmented */ 
                
            }


    }

    