/* Driver For PCF8583T Real Time Clock IC */
/* ************************************** */


/* Version 1.00 */
/* Start Date: 23/6/2009 */
/* The RTC Clock IC is part of the communications module */





/* Include Files Here */
#include "mbed.h"                                /* mbed header file */
#include "misra_types.h"                         /* MISRA Types header file */
#include "defines.h"
#include "RSEDP_Slave_Address_Defines.h"        /* Slave Address of CNTRL I2C devices */

#include "mbed_Port_Structure.h"                /* Port structure for MBED Module */
#include "RSEDP_CNTRL_I2C.h"                    /* Control I2C Driver */



#define START_YEAR (uint16_t) 2009                /* Value used for real time clock */



                                        
/* Function Prototypes Here */
sint32_t RSEDP_COM_setup_PCF8583(uint8_t Slave_Address, uint8_t mode);                                                  /* Setup the device */
sint32_t RSEDP_COM_PCF8583_Read_Status_Register(uint8_t Slave_Address, uint8_t *status_register);                       /* Read the status register of the device */
sint32_t RSEDP_COM_PCF8583_Write_Status_Register(uint8_t Slave_Address, uint8_t payload);                               /* Status register write */

/* RTC Function */
sint32_t RSEDP_COM_PCF8583_Write_RTC(uint8_t Slave_Address, uint8_t *RTC_Array);                                        /* Write raw data to the RTC device */
sint32_t RSEDP_COM_PCF8583_Read_RTC(uint8_t Slave_Address, uint8_t *RTC_Array);                                         /* Read raw data from the real time clock */
sint32_t RSEDP_COM_PCF8583_Set_Clock(uint8_t Slave_Address, uint8_t hours, uint8_t minutes, uint8_t seconds,uint8_t F24hour_flag, uint8_t AMPM_flag);    /* Set the clock only */
sint32_t RSEDP_COM_PCF8583_Set_Date(uint8_t Slave_Address, uint8_t dow, uint8_t day,uint8_t month, uint16_t year);      /* Set the date only */
sint32_t RSEDP_COM_PCF8583_Read_Clock(uint8_t Slave_Address, uint8_t *Time_Array);                                      /* Read the time component only */
sint32_t RSEDP_COM_PCF8583_Read_Date(uint8_t Slave_Address, uint8_t *Date_Array);                                       /* Read the date component only */
sint32_t RSEDP_COM_PCF8583_Read_Clock_And_Date(uint8_t Slave_Address, uint8_t *Time_Array,uint8_t *Date_Array);         /* Read time and date */ 
sint32_t RSEDP_COM_PCF8583_Start_Clock(uint8_t Slave_Address);                                                          /* Start the clock counting */
sint32_t RSEDP_COM_PCF8583_Stop_Clock(uint8_t Slave_Address);                                                           /* Stop the clock */

/* Battery Maintained SRAM functions */
sint32_t RSEDP_COM_PCF8583_Read_SRAM(uint8_t Slave_Address, uint8_t *read_data,uint8_t address);                        /* Read data from the battery maintained SRAM */
sint32_t RSEDP_COM_PCF8583_Write_SRAM(uint8_t Slave_Address, uint8_t payload,uint8_t address);                          /* Write data to the battery maintained SRAM */


/* Print Functions */
void RSEDP_COM_Print_Time(uint8_t *Time_Array);
void RSEDP_COM_Print_Date(uint8_t *Date_Array);
void RSEDP_COM_Print_Time_Date_Year(uint8_t *RTC_Array);



/* Static local functions */
static void delay_small(void);





/* Setup the device */
sint32_t RSEDP_COM_setup_PCF8583(uint8_t Slave_Address, uint8_t mode)
    {
        sint32_t Ack_Status = 0;
        /* Assume the I2C Peripheral is already configured */

        if ((mode=='c') | (mode== 'C')) 
            {
                /* Configure for Clock mode */
                Ack_Status = RSEDP_COM_PCF8583_Write_Status_Register(Slave_Address, 0x00);
        
            }
    
        if ((mode=='e') | (mode== 'E')) 
            {
                /* configure for Event Counter Mode */
                Ack_Status = RSEDP_COM_PCF8583_Write_Status_Register(Slave_Address, 0x20);
            } 
        return Ack_Status;
    }




/* Read the status register of the device */
sint32_t RSEDP_COM_PCF8583_Read_Status_Register(uint8_t Slave_Address, uint8_t *status_register)
    {
        sint32_t Ack_Status = 0;
        sint8_t tx_array[2];
        sint8_t rx_array=0;                                                        /* Variable used to receive data from slave */

        tx_array[0] = 0x00;                                                        /* Memory address 0x00 */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 1);   /* Set the RTC address */

        if (Ack_Status == ACK)
            {
                Ack_Status = CNTRL_I2C_Master_Mode_Receive(Slave_Address,  &rx_array,1);                /* Read single byte address */
            }
        *status_register = rx_array;                                               /* Load value from RTC into receive pointer */
        return Ack_Status;
    }    




/* Status register write */
sint32_t RSEDP_COM_PCF8583_Write_Status_Register(uint8_t Slave_Address, uint8_t payload)
    {
        sint32_t Ack_Status = 0;
        sint8_t tx_array[2];                                                           /* Transmission array */

        tx_array[0] = 0x00;                                                            /* Address 0x00 is where the control register is located */
        tx_array[1] = payload;                                                         /* Data to be written */

        Ack_Status  = CNTRL_I2C_Master_Mode_Transmit(Slave_Address,  tx_array,2);                   /* Send the data */
        return Ack_Status;
    }


/* Read the real time clock information */
sint32_t RSEDP_COM_PCF8583_Read_RTC(uint8_t Slave_Address, uint8_t *RTC_Array)
    {                        
        sint32_t Ack_Status = 0;
        sint8_t tx_array[2];
        sint8_t local_array[8];    
        uint8_t n = 0u;

        tx_array[0] = 0x01;                                                        /* Memory address 0x01 */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 1);                /* Set the RTC memory address */

        if (Ack_Status == ACK)
            {
                Ack_Status = CNTRL_I2C_Master_Mode_Receive(Slave_Address, local_array,7);            /* Read single byte address */
            }
        for (n = 0; n < 7; n++)
            {
                RTC_Array[n] = (uint8_t) (local_array[n]);
            }
        return Ack_Status;
    }


/* Write the real time clock information */
sint32_t RSEDP_COM_PCF8583_Write_RTC(uint8_t Slave_Address, uint8_t *RTC_Array)                    
     {                        
        sint32_t Ack_Status = 0;
        sint8_t tx_array[8];
        uint8_t n = 0;

        tx_array[0] = 0x01;                                                        /* Memory address 0x01 */
        for (n = 1; n < 8; n++)                                                    /* Copy data into array */    
            {
                tx_array[n] = *RTC_Array;
                RTC_Array++;
            }

        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 8);   /* Transmit the data */
        delay_small();                                                             /* RTC is busy writing */
        return Ack_Status;
    }


/* Set the clock inside the device */
sint32_t RSEDP_COM_PCF8583_Set_Clock(uint8_t Slave_Address, uint8_t hours, uint8_t minutes, uint8_t seconds,uint8_t F24hour_flag, uint8_t AMPM_flag)
    {
        sint32_t Ack_Status = 0;
        sint8_t RTC_Array[5] = {0,0,0,0,0};
        uint8_t high = 0;
        uint8_t low = 0;

           /* Hundreths */
        RTC_Array[1] = 0x00;                                                    /* Hundreths of a second -set to zero */
           
        /* Seconds */                                                            /* Convert to BCD */
        low = (seconds % 10);
        high = (seconds / 10);
        RTC_Array[2] = ((high << 4) + low);
        
        /* Minutes */                                                             /* Convert to BCD */
        low = (minutes % 10);
        high = (minutes / 10);
        RTC_Array[3] = ((high << 4) + low);

        /* Hours */                                                                /* Convert to BCD  and add in 24 hour and AM/PM flag */
        low = (hours % 10);
        high = (hours / 10);
        RTC_Array[4] = ((high << 4) + low);
        RTC_Array[4] = (RTC_Array[4] & 0x3f);

           F24hour_flag = ((F24hour_flag & 0x01) << 7);
        AMPM_flag = ((AMPM_flag & 0x01) << 6);

        RTC_Array[4] += F24hour_flag + AMPM_flag;
    
        RTC_Array[0] = 0x01;                                                    /* Target destination address of data */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, RTC_Array, 5);    /* Transmit the data */
        delay_small();                                                             /* RTC is busy writing */
        return Ack_Status;
    }



/* Write the time and date to the RTC clock */
sint32_t RSEDP_COM_PCF8583_Set_Date(uint8_t Slave_Address, uint8_t dow, uint8_t day,uint8_t month, uint16_t year)
    {
        sint32_t Ack_Status = 0;
        sint8_t RTC_Array[5] = {0,0,0,0,0};
        uint8_t high = 0;
        uint8_t low = 0;

        /* day of month */                                                            /* Convert to BCD */
        if (day > 31) day = 31;
        low = (day % 10);
        high = (day / 10);
        RTC_Array[1] = (((high << 4) + low) & 0x3f);

        /* year */
        year = ((year - 2008) & 0x03);
        RTC_Array[1] += (year << 6);

        /* Day of week */
           if (dow > 6) dow = 6;
        dow = (dow << 5);
        RTC_Array[2] = dow;

        /* months */
        if (month > 12) month = 12;
        low = (month % 10);
        high = (month / 10);
        RTC_Array[2] += (((high << 4) + low));

            
        RTC_Array[0] = 0x05;                                                          /* Target destination address of data */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, RTC_Array, 3);     /* Transmit the data */
        delay_small();                                                                /* RTC is busy writing */
        return Ack_Status;
    }



/* Start the clock counting */
sint32_t RSEDP_COM_PCF8583_Start_Clock(uint8_t Slave_Address)
    {
        sint32_t Ack_Status = 0;
        uint8_t status_reg = 0;
        
        Ack_Status = RSEDP_COM_PCF8583_Read_Status_Register(Slave_Address, &status_reg);
    
        status_reg = (status_reg & 0x7f);
    
        if (Ack_Status ==ACK)
            {
                Ack_Status = RSEDP_COM_PCF8583_Write_Status_Register(Slave_Address, status_reg);
            }
        return Ack_Status;
    }



/* Stop the clock */
sint32_t RSEDP_COM_PCF8583_Stop_Clock(uint8_t Slave_Address)
    {
        sint32_t Ack_Status = 0;
        uint8_t status_reg = 0;
        
        Ack_Status = RSEDP_COM_PCF8583_Read_Status_Register(Slave_Address, &status_reg);
    
        status_reg = (status_reg | 0x80);
        
        if (Ack_Status == ACK)
            {
                RSEDP_COM_PCF8583_Write_Status_Register(Slave_Address, status_reg);
            }
        return Ack_Status;
    }


/* Read the time component only */
sint32_t RSEDP_COM_PCF8583_Read_Clock(uint8_t Slave_Address, uint8_t *Time_Array)                          
    {
        sint32_t Ack_Status = 0;
        uint8_t RTC_Array[6] = {0,0,0,0,0,0};                                        /* Hundreths, seconds, minutes,hours, year/date, weekday/month */
        uint8_t temp8 = 0;    

        Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);

        /* hours */
        temp8 = (((RTC_Array[3] & 0x30) >> 4) * 10);
        *Time_Array = ((RTC_Array[3] & 0x0f) + temp8);
        Time_Array++;
                
        /* Minutes */
        temp8=(((RTC_Array[2] & 0xf0) >> 4) * 10);
           *Time_Array = ((RTC_Array[2] & 0x0f) + temp8);
        Time_Array++;

        /* Seconds */
        temp8=(((RTC_Array[1] & 0xf0) >> 4) * 10);
           *Time_Array = ((RTC_Array[1] & 0x0f) + temp8);
        Time_Array++;

        /* Hundreths */
        temp8=(((RTC_Array[0] & 0xf0) >> 4) * 10);
           *Time_Array = ((RTC_Array[0] & 0x0f) + temp8);
        Time_Array++;

        // 24 Hour Flag
        *Time_Array = ((RTC_Array[3] & 0x80) >> 7);
        Time_Array++;
                
        // AMPM Flag
        *Time_Array = (( RTC_Array[3] & 0x40) >> 6);
        return Ack_Status;
    }                    



/* Read the date component only */
sint32_t RSEDP_COM_PCF8583_Read_Date(uint8_t Slave_Address, uint8_t *Date_Array)
    {
        sint32_t Ack_Status = 0;
        uint8_t RTC_Array[6] = {0,0,0,0,0,0};                                        /* Hundreths, seconds, minutes,hours, year/date, weekday/month */
        uint8_t temp8 = 0;    

        Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);

        /* Weekday */
        *Date_Array = ((RTC_Array[5] & 0xe0) >> 5);
        Date_Array++;        

        /* Date */
        temp8=(((RTC_Array[4] & 0x30) >> 4) * 10);
        *Date_Array = ((RTC_Array[4] & 0x0f) + temp8);
        Date_Array++;
                        
        /* Month */
        temp8 = (((RTC_Array[5] & 0x10) >> 4) * 10);
        *Date_Array = ((RTC_Array[5] & 0x0f) + temp8);    
        Date_Array++;

        /* Year */
        *Date_Array = ((RTC_Array[4] & 0xc0) >> 6);
        return Ack_Status;
    }


/* Read time and date */ 
sint32_t RSEDP_COM_PCF8583_Read_Clock_And_Date(uint8_t Slave_Address, uint8_t *Time_Array,uint8_t *Date_Array) 
    {
        sint32_t Ack_Status = 0;
        uint8_t RTC_Array[6] = {0,0,0,0,0,0};                                        /* Hundreths, seconds, minutes,hours, year/date, weekday/month */
        uint8_t temp8 = 0;    

        Ack_Status = RSEDP_COM_PCF8583_Read_RTC(Slave_Address, RTC_Array);

        /* hours */
        temp8 = (((RTC_Array[3] & 0x30) >> 4) * 10);
        *Time_Array = ((RTC_Array[3] & 0x0f) + temp8);
        Time_Array++;
                
        /* Minutes */
        temp8 = (((RTC_Array[2] & 0xf0) >> 4) * 10);
        *Time_Array = ((RTC_Array[2] & 0x0f) + temp8);
        Time_Array++;

        /* Seconds */
        temp8 = (((RTC_Array[1] & 0xf0) >> 4) * 10);
           *Time_Array = ((RTC_Array[1] & 0x0f) + temp8);
        Time_Array++;

        /* Hundreths */
        temp8 = (((RTC_Array[0] & 0xf0) >> 4) * 10);
           *Time_Array = ((RTC_Array[0] & 0x0f) + temp8);
        Time_Array++;

        /* 24 Hour Flag */
        *Time_Array = ((RTC_Array[3] & 0x80)>>7);
        Time_Array++;
                
        /* AMPM Flag */
        *Time_Array = (( RTC_Array[3] & 0x40) >> 6);

        
        /* Date */
        /* Weekday */
        *Date_Array = ((RTC_Array[5] & 0xe0) >> 5);
        Date_Array++;        

        /* Date */
        temp8 = (((RTC_Array[4] & 0x30) >> 4) * 10);
        *Date_Array = ((RTC_Array[4] & 0x0f) + temp8);
        Date_Array++;
                        
        /* Month */
        temp8 = (((RTC_Array[5] & 0x10) >> 4) * 10);
        *Date_Array = ((RTC_Array[5] & 0x0f) + temp8);    
        Date_Array++;

        /* Year */
        *Date_Array = ((RTC_Array[4] & 0xc0) >> 6);
        return Ack_Status;
    }







/* Battery Maintained SRAM functions */

/* Read data from the battery maintained SRAM */
sint32_t RSEDP_COM_PCF8583_Read_SRAM(uint8_t Slave_Address, uint8_t *read_data, uint8_t address)    
    {        
        sint32_t Ack_Status = 0;
        sint8_t tx_array[2];                                                    /* Transmission array of data */
        sint8_t rx_array=0;                                                        /* Variable used to receive data from slave */
    
        tx_array[0] = address;                                                    /* Memory address to read */
        
        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array, 1);                /* Set the RTC address */

        if (Ack_Status == ACK)
            {
                CNTRL_I2C_Master_Mode_Receive(Slave_Address, &rx_array,1);                /* Read single byte address */
            }
        *read_data=rx_array;                                                   /* Load value from RTC into receive pointer */
        return Ack_Status;
    }


/* Write data to the battery maintained SRAM */
sint32_t RSEDP_COM_PCF8583_Write_SRAM(uint8_t Slave_Address, uint8_t payload, uint8_t address)
    {
        sint32_t Ack_Status = 0;
        sint8_t tx_array[2];                                                     /* Transmission array */

        tx_array[0] = address;                                                    /* Address */
        tx_array[1] = payload;                                                    /* Data */ 

        Ack_Status = CNTRL_I2C_Master_Mode_Transmit(Slave_Address, tx_array,2);                   /* Send the data */
        return Ack_Status;

    }




/* Print the time, date and year */
void RSEDP_COM_Print_Time_Date_Year(uint8_t *RTC_Array)
    {
        uint8_t hundreths = 0;
        uint8_t seconds = 0;
        uint8_t minutes = 0;
        uint8_t hours = 0;
        uint8_t date = 0;
        uint16_t year = 0;
        uint8_t weekday = 0;
        uint8_t month = 0;
        uint8_t temp8 = 0;
        uint8_t AMPM_flag = 0;
        uint8_t F24hour_flag=0;

        /* Hundreths */
        temp8 = (((*RTC_Array & 0xf0) >> 4) * 10);
        hundreths = ((*RTC_Array & 0x0f) + temp8);
        RTC_Array++;

        /* Seconds */
        temp8 = (((*RTC_Array & 0xf0) >> 4) * 10);
        seconds = ((*RTC_Array & 0x0f) + temp8);
        RTC_Array++;

        /* Minutes */
        temp8 = (((*RTC_Array & 0xf0) >> 4) * 10);
        minutes = ((*RTC_Array & 0x0f) + temp8);
        RTC_Array++;

        /* hours */
        temp8 = (((*RTC_Array & 0x30) >> 4) * 10);
        hours = ((*RTC_Array & 0x0f) + temp8);
        F24hour_flag = ((*RTC_Array & 0x80) >> 7);
        AMPM_flag = (( *RTC_Array & 0x40) >> 6);
        RTC_Array++;

        /* Date */
        temp8 = (((*RTC_Array & 0x30) >> 4) * 10);
        date = ((*RTC_Array & 0x0f) + temp8);
        
        /* Year */
        year = ((*RTC_Array & 0xc0) >> 6);
        RTC_Array++;

        /* Weekday */
        weekday = ((*RTC_Array & 0xe0) >> 5);
        
        /* Month */
        temp8 = (((*RTC_Array & 0x10) >> 4) * 10);
        month = ((*RTC_Array & 0x0f) + temp8);    
                                  
        pc.printf("%.2d:", hours);
        pc.printf("%.2d ", minutes); 

        if (F24hour_flag==1)    
                {
                    if (AMPM_flag == 0) 
                        {
                            pc.printf("AM ");
                        }
                    if (AMPM_flag == 1) 
                        {
                            pc.printf("PM ");
                        }
                }

        pc.printf("%.2d.",seconds);
        pc.printf("%.2d seconds ", hundreths); 
        
        if (weekday == 0) pc.printf("Monday");        
        if (weekday == 1) pc.printf("Tuesday");        
        if (weekday == 2) pc.printf("Wednesday");        
        if (weekday == 3) pc.printf("Thursday");        
        if (weekday == 4) pc.printf("Friday");        
        if (weekday == 5) pc.printf("Saturday");        
        if (weekday == 6) pc.printf("Sunday");        
        
        pc.printf(" %.2d ",date);
        
        if (month == 1) pc.printf("January");
        if (month == 2) pc.printf("Febuary");
        if (month == 3) pc.printf("March");
        if (month == 4) pc.printf("April");
        if (month == 5) pc.printf("May");
        if (month == 6) pc.printf("June");
        if (month == 7) pc.printf("July");
        if (month == 8) pc.printf("August");
        if (month == 9) pc.printf("September");
        if (month == 10) pc.printf("October");
        if (month == 11) pc.printf("November");
        if (month == 12) pc.printf("December");
    
           year = year + START_YEAR;
        pc.printf(" %.4d       ", year);


    }

/* Print the Time */
void RSEDP_COM_Print_Time(uint8_t *Time_Array)
    {
        uint8_t hours = 0;
        uint8_t minutes = 0;
        uint8_t seconds = 0;
        uint8_t hundreths = 0;
        uint8_t F24hour_flag=0;
        uint8_t AMPM_flag = 0;

        hours = *Time_Array;
        Time_Array++;
        minutes = *Time_Array;
        Time_Array++;    
        seconds = *Time_Array;
        Time_Array++;
        hundreths = *Time_Array;
        Time_Array++;
        F24hour_flag = *Time_Array;
        Time_Array++;    
        AMPM_flag = *Time_Array;
        
        pc.printf("%.2d:",hours);
        pc.printf("%.2d ",minutes); 

        if (F24hour_flag == 1)    
                {
                    if (AMPM_flag == 0) 
                        {
                            pc.printf("AM ");
                        }
                    if (AMPM_flag == 1) 
                        {
                            pc.printf("PM ");
                        }
                }

        pc.printf("%.2d.",seconds);
        pc.printf("%.2d seconds ",hundreths); 

    }


/* Print the date */
void RSEDP_COM_Print_Date(uint8_t *Date_Array)
    {
        uint8_t date = 0;
        uint16_t year = 0;
        uint8_t weekday = 0;
        uint8_t month = 0;
    
        weekday = *Date_Array;
        Date_Array++;
        date = *Date_Array;
        Date_Array++;
        month = *Date_Array;
        Date_Array++;
        year = *Date_Array;
        Date_Array++;

        if (weekday == 0) pc.printf("Monday");        
        if (weekday == 1) pc.printf("Tuesday");        
        if (weekday == 2) pc.printf("Wednesday");        
        if (weekday == 3) pc.printf("Thursday");        
        if (weekday == 4) pc.printf("Friday");        
        if (weekday == 5) pc.printf("Saturday");        
        if (weekday == 6) pc.printf("Sunday");        
        
        pc.printf(" %.2d ",date);
        
        if (month == 1) pc.printf("January");
        if (month == 2) pc.printf("Febuary");
        if (month == 3) pc.printf("March");
        if (month == 4) pc.printf("April");
        if (month == 5) pc.printf("May");
        if (month == 6) pc.printf("June");
        if (month == 7) pc.printf("July");
        if (month == 8) pc.printf("August");
        if (month == 9) pc.printf("September");
        if (month == 10) pc.printf("October");
        if (month == 11) pc.printf("November");
        if (month == 12) pc.printf("December");
    
           year = year + START_YEAR;
        pc.printf(" %.4d       ", year);
    }











/* Small 5ms delay approx */
/* Note: Change this delay to suite the write time/busy period of the RTC */
static void delay_small(void)
    {
        uint32_t nnnn=0;

        for(nnnn = 0; nnnn < 0x12000; nnnn++)
            {
                ;
            }
    }
